/*
 * Decompiled with CFR 0.152.
 */
package fr.ax_dev.universeProfiles.database;

import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;
import fr.ax_dev.universeProfiles.UniverseProfiles;
import fr.ax_dev.universeProfiles.database.DatabaseType;
import fr.ax_dev.universeProfiles.models.PlayerProfile;
import fr.ax_dev.universeProfiles.utils.ItemSerializer;
import java.io.File;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.bukkit.configuration.file.FileConfiguration;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.JedisPoolConfig;

public class DatabaseManager {
    private final UniverseProfiles plugin;
    private final DatabaseType type;
    private HikariDataSource dataSource;
    private JedisPool jedisPool;

    public DatabaseManager(UniverseProfiles plugin, DatabaseType type) {
        this.plugin = plugin;
        this.type = type;
    }

    public void initialize() {
        FileConfiguration config = this.plugin.getConfigManager().getConfig();
        switch (this.type) {
            case SQLITE: {
                this.initializeSQLite();
                break;
            }
            case MYSQL: {
                this.initializeMySQL(config);
                break;
            }
            case REDIS: {
                this.initializeRedis(config);
            }
        }
        if (this.type != DatabaseType.REDIS) {
            this.createTables();
        }
    }

    private void initializeSQLite() {
        HikariConfig hikariConfig = new HikariConfig();
        File dbFile = new File(this.plugin.getDataFolder(), "profiles.db");
        hikariConfig.setJdbcUrl("jdbc:sqlite:" + dbFile.getAbsolutePath());
        hikariConfig.setDriverClassName("org.sqlite.JDBC");
        hikariConfig.setMaximumPoolSize(10);
        hikariConfig.setConnectionTimeout(30000L);
        this.dataSource = new HikariDataSource(hikariConfig);
    }

    private void initializeMySQL(FileConfiguration config) {
        HikariConfig hikariConfig = new HikariConfig();
        String host = config.getString("database.mysql.host", "localhost");
        int port = config.getInt("database.mysql.port", 3306);
        String database = config.getString("database.mysql.database", "universe_profiles");
        String username = config.getString("database.mysql.username", "root");
        String password = config.getString("database.mysql.password", "");
        hikariConfig.setJdbcUrl("jdbc:mysql://" + host + ":" + port + "/" + database + "?useSSL=false&autoReconnect=true");
        hikariConfig.setUsername(username);
        hikariConfig.setPassword(password);
        hikariConfig.setDriverClassName("com.mysql.cj.jdbc.Driver");
        hikariConfig.setMaximumPoolSize(10);
        hikariConfig.setConnectionTimeout(30000L);
        hikariConfig.addDataSourceProperty("cachePrepStmts", "true");
        hikariConfig.addDataSourceProperty("prepStmtCacheSize", "250");
        hikariConfig.addDataSourceProperty("prepStmtCacheSqlLimit", "2048");
        this.dataSource = new HikariDataSource(hikariConfig);
    }

    private void initializeRedis(FileConfiguration config) {
        String host = config.getString("database.redis.host", "localhost");
        int port = config.getInt("database.redis.port", 6379);
        String password = config.getString("database.redis.password", "");
        JedisPoolConfig poolConfig = new JedisPoolConfig();
        poolConfig.setMaxTotal(10);
        poolConfig.setMaxIdle(5);
        poolConfig.setMinIdle(1);
        this.jedisPool = password != null && !password.isEmpty() ? new JedisPool((GenericObjectPoolConfig<Jedis>)poolConfig, host, port, 2000, password) : new JedisPool((GenericObjectPoolConfig<Jedis>)poolConfig, host, port, 2000);
    }

    private void createTables() {
        String createProfilesTable = "CREATE TABLE IF NOT EXISTS profiles (\n    uuid VARCHAR(36) PRIMARY KEY,\n    player_name VARCHAR(16),\n    is_public BOOLEAN DEFAULT TRUE,\n    stars INTEGER DEFAULT 0,\n    star_givers TEXT,\n    inventory TEXT,\n    armor TEXT,\n    main_hand TEXT,\n    off_hand TEXT,\n    last_seen BIGINT\n)\n";
        try (Connection conn = this.dataSource.getConnection();
             Statement stmt = conn.createStatement();){
            stmt.execute(createProfilesTable);
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to create database tables: " + e.getMessage());
        }
    }

    public CompletableFuture<PlayerProfile> loadProfile(UUID uuid) {
        return CompletableFuture.supplyAsync(() -> {
            if (this.type == DatabaseType.REDIS) {
                return this.loadFromRedis(uuid);
            }
            return this.loadFromSQL(uuid);
        });
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private PlayerProfile loadFromSQL(UUID uuid) {
        String query = "SELECT * FROM profiles WHERE uuid = ?";
        try (Connection conn = this.dataSource.getConnection();
             PreparedStatement stmt = conn.prepareStatement(query);){
            String offHandStr;
            String mainHandStr;
            String armorStr;
            String inventoryStr;
            stmt.setString(1, uuid.toString());
            ResultSet rs = stmt.executeQuery();
            if (!rs.next()) return null;
            PlayerProfile profile = new PlayerProfile(uuid, rs.getString("player_name"));
            profile.setPublic(rs.getBoolean("is_public"));
            profile.setStars(rs.getInt("stars"));
            String starGiversStr = rs.getString("star_givers");
            if (starGiversStr != null && !starGiversStr.isEmpty()) {
                HashSet<UUID> starGivers = new HashSet<UUID>();
                for (String uuidStr : starGiversStr.split(",")) {
                    starGivers.add(UUID.fromString(uuidStr));
                }
                profile.setStarGivers(starGivers);
            }
            if ((inventoryStr = rs.getString("inventory")) != null) {
                profile.setInventory(ItemSerializer.deserializeInventory(inventoryStr));
            }
            if ((armorStr = rs.getString("armor")) != null) {
                profile.setArmorContents(ItemSerializer.deserializeArmor(armorStr));
            }
            if ((mainHandStr = rs.getString("main_hand")) != null) {
                profile.setMainHand(ItemSerializer.deserializeItem(mainHandStr));
            }
            if ((offHandStr = rs.getString("off_hand")) != null) {
                profile.setOffHand(ItemSerializer.deserializeItem(offHandStr));
            }
            profile.setLastSeen(rs.getLong("last_seen"));
            PlayerProfile playerProfile = profile;
            return playerProfile;
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to load profile from database: " + e.getMessage());
        }
        return null;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private PlayerProfile loadFromRedis(UUID uuid) {
        try (Jedis jedis = this.jedisPool.getResource();){
            String offHandStr;
            String mainHandStr;
            String armorStr;
            String inventoryStr;
            Map<String, String> data = jedis.hgetAll("profile:" + uuid.toString());
            if (data.isEmpty()) return null;
            PlayerProfile profile = new PlayerProfile(uuid, data.get("player_name"));
            profile.setPublic(Boolean.parseBoolean(data.get("is_public")));
            profile.setStars(Integer.parseInt(data.getOrDefault("stars", "0")));
            String starGiversStr = data.get("star_givers");
            if (starGiversStr != null && !starGiversStr.isEmpty()) {
                HashSet<UUID> starGivers = new HashSet<UUID>();
                for (String uuidStr : starGiversStr.split(",")) {
                    starGivers.add(UUID.fromString(uuidStr));
                }
                profile.setStarGivers(starGivers);
            }
            if ((inventoryStr = data.get("inventory")) != null) {
                profile.setInventory(ItemSerializer.deserializeInventory(inventoryStr));
            }
            if ((armorStr = data.get("armor")) != null) {
                profile.setArmorContents(ItemSerializer.deserializeArmor(armorStr));
            }
            if ((mainHandStr = data.get("main_hand")) != null) {
                profile.setMainHand(ItemSerializer.deserializeItem(mainHandStr));
            }
            if ((offHandStr = data.get("off_hand")) != null) {
                profile.setOffHand(ItemSerializer.deserializeItem(offHandStr));
            }
            profile.setLastSeen(Long.parseLong(data.getOrDefault("last_seen", "0")));
            PlayerProfile playerProfile = profile;
            return playerProfile;
        }
        catch (Exception e) {
            this.plugin.getLogger().severe("Failed to load profile from Redis: " + e.getMessage());
        }
        return null;
    }

    public CompletableFuture<Void> saveProfile(PlayerProfile profile) {
        return CompletableFuture.runAsync(() -> {
            if (this.type == DatabaseType.REDIS) {
                this.saveToRedis(profile);
            } else {
                this.saveToSQL(profile);
            }
        });
    }

    private void saveToSQL(PlayerProfile profile) {
        String query = "INSERT INTO profiles (uuid, player_name, is_public, stars, star_givers, inventory, armor, main_hand, off_hand, last_seen)\nVALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)\nON CONFLICT(uuid) DO UPDATE SET\n    player_name = excluded.player_name,\n    is_public = excluded.is_public,\n    stars = excluded.stars,\n    star_givers = excluded.star_givers,\n    inventory = excluded.inventory,\n    armor = excluded.armor,\n    main_hand = excluded.main_hand,\n    off_hand = excluded.off_hand,\n    last_seen = excluded.last_seen\n";
        if (this.type == DatabaseType.MYSQL) {
            query = query.replace("ON CONFLICT(uuid) DO UPDATE SET", "ON DUPLICATE KEY UPDATE");
            query = query.replace("excluded.", "VALUES(").replaceAll("excluded\\.([a-z_]+)", "VALUES($1)");
        }
        try (Connection conn = this.dataSource.getConnection();
             PreparedStatement stmt = conn.prepareStatement(query);){
            stmt.setString(1, profile.getUuid().toString());
            stmt.setString(2, profile.getPlayerName());
            stmt.setBoolean(3, profile.isPublic());
            stmt.setInt(4, profile.getStars());
            String starGivers = String.join((CharSequence)",", profile.getStarGivers().stream().map(UUID::toString).toList());
            stmt.setString(5, starGivers);
            stmt.setString(6, ItemSerializer.serializeInventory(profile.getInventory()));
            stmt.setString(7, ItemSerializer.serializeArmor(profile.getArmorContents()));
            stmt.setString(8, ItemSerializer.serializeItem(profile.getMainHand()));
            stmt.setString(9, ItemSerializer.serializeItem(profile.getOffHand()));
            stmt.setLong(10, profile.getLastSeen());
            stmt.executeUpdate();
        }
        catch (SQLException e) {
            this.plugin.getLogger().severe("Failed to save profile to database: " + e.getMessage());
        }
    }

    private void saveToRedis(PlayerProfile profile) {
        try (Jedis jedis = this.jedisPool.getResource();){
            HashMap<String, String> data = new HashMap<String, String>();
            data.put("player_name", profile.getPlayerName());
            data.put("is_public", String.valueOf(profile.isPublic()));
            data.put("stars", String.valueOf(profile.getStars()));
            String starGivers = String.join((CharSequence)",", profile.getStarGivers().stream().map(UUID::toString).toList());
            data.put("star_givers", starGivers);
            data.put("inventory", ItemSerializer.serializeInventory(profile.getInventory()));
            data.put("armor", ItemSerializer.serializeArmor(profile.getArmorContents()));
            data.put("main_hand", ItemSerializer.serializeItem(profile.getMainHand()));
            data.put("off_hand", ItemSerializer.serializeItem(profile.getOffHand()));
            data.put("last_seen", String.valueOf(profile.getLastSeen()));
            jedis.hset("profile:" + profile.getUuid().toString(), data);
        }
        catch (Exception e) {
            this.plugin.getLogger().severe("Failed to save profile to Redis: " + e.getMessage());
        }
    }

    public void close() {
        if (this.dataSource != null) {
            this.dataSource.close();
        }
        if (this.jedisPool != null) {
            this.jedisPool.close();
        }
    }
}

