/*
 * Decompiled with CFR 0.152.
 */
package io.jsonwebtoken.impl.security;

import io.jsonwebtoken.impl.lang.Bytes;
import io.jsonwebtoken.impl.lang.Function;
import io.jsonwebtoken.impl.security.AbstractCurve;
import io.jsonwebtoken.impl.security.DefaultKeyPairBuilder;
import io.jsonwebtoken.impl.security.EdwardsPublicKeyDeriver;
import io.jsonwebtoken.impl.security.JcaTemplate;
import io.jsonwebtoken.impl.security.KeysBridge;
import io.jsonwebtoken.impl.security.NamedParameterSpecValueFinder;
import io.jsonwebtoken.lang.Assert;
import io.jsonwebtoken.lang.Collections;
import io.jsonwebtoken.lang.Strings;
import io.jsonwebtoken.security.InvalidKeyException;
import io.jsonwebtoken.security.KeyException;
import io.jsonwebtoken.security.KeyLengthSupplier;
import io.jsonwebtoken.security.KeyPairBuilder;
import java.security.Key;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.spec.KeySpec;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.Map;

public class EdwardsCurve
extends AbstractCurve
implements KeyLengthSupplier {
    private static final String OID_PREFIX = "1.3.101.";
    private static final byte[] ASN1_OID_PREFIX = new byte[]{6, 3, 43, 101};
    private static final Function<Key, String> CURVE_NAME_FINDER = new NamedParameterSpecValueFinder();
    public static final EdwardsCurve X25519 = new EdwardsCurve("X25519", 110);
    public static final EdwardsCurve X448 = new EdwardsCurve("X448", 111);
    public static final EdwardsCurve Ed25519 = new EdwardsCurve("Ed25519", 112);
    public static final EdwardsCurve Ed448 = new EdwardsCurve("Ed448", 113);
    public static final Collection<EdwardsCurve> VALUES = Collections.of(X25519, X448, Ed25519, Ed448);
    private static final Map<String, EdwardsCurve> REGISTRY = new LinkedHashMap<String, EdwardsCurve>(8);
    private static final Map<Integer, EdwardsCurve> BY_OID_TERMINAL_NODE = new LinkedHashMap<Integer, EdwardsCurve>(4);
    private final String OID;
    final byte[] ASN1_OID;
    private final int keyBitLength;
    private final int encodedKeyByteLength;
    private final byte[] PUBLIC_KEY_ASN1_PREFIX;
    private final byte[] PRIVATE_KEY_ASN1_PREFIX;
    private final byte[] PRIVATE_KEY_JDK11_PREFIX;
    private final boolean signatureCurve;

    private static byte[] publicKeyAsn1Prefix(int byteLength, byte[] ASN1_OID) {
        return Bytes.concat({48, (byte)(byteLength + 10), 48, 5}, ASN1_OID, {3, (byte)(byteLength + 1), 0});
    }

    private static byte[] privateKeyPkcs8Prefix(int byteLength, byte[] ASN1_OID, boolean ber) {
        byte[] byArray;
        if (ber) {
            byte[] byArray2 = new byte[4];
            byArray2[0] = 4;
            byArray2[1] = (byte)(byteLength + 2);
            byArray2[2] = 4;
            byArray = byArray2;
            byArray2[3] = (byte)byteLength;
        } else {
            byte[] byArray3 = new byte[2];
            byArray3[0] = 4;
            byArray = byArray3;
            byArray3[1] = (byte)byteLength;
        }
        byte[] keyPrefix = byArray;
        return Bytes.concat({48, (byte)(5 + ASN1_OID.length + keyPrefix.length + byteLength), 2, 1, 0, 48, 5}, ASN1_OID, keyPrefix);
    }

    EdwardsCurve(String id, int oidTerminalNode) {
        super(id, id);
        if (oidTerminalNode < 110 || oidTerminalNode > 113) {
            String msg = "Invalid Edwards Curve ASN.1 OID terminal node value";
            throw new IllegalArgumentException(msg);
        }
        this.keyBitLength = oidTerminalNode % 2 == 0 ? 255 : 448;
        int encodingBitLen = oidTerminalNode == 113 ? this.keyBitLength + 8 : this.keyBitLength;
        this.encodedKeyByteLength = Bytes.length(encodingBitLen);
        this.OID = OID_PREFIX + oidTerminalNode;
        this.signatureCurve = oidTerminalNode == 112 || oidTerminalNode == 113;
        byte[] suffix = new byte[]{(byte)oidTerminalNode};
        this.ASN1_OID = Bytes.concat(ASN1_OID_PREFIX, suffix);
        this.PUBLIC_KEY_ASN1_PREFIX = EdwardsCurve.publicKeyAsn1Prefix(this.encodedKeyByteLength, this.ASN1_OID);
        this.PRIVATE_KEY_ASN1_PREFIX = EdwardsCurve.privateKeyPkcs8Prefix(this.encodedKeyByteLength, this.ASN1_OID, true);
        this.PRIVATE_KEY_JDK11_PREFIX = EdwardsCurve.privateKeyPkcs8Prefix(this.encodedKeyByteLength, this.ASN1_OID, false);
    }

    @Override
    public int getKeyBitLength() {
        return this.keyBitLength;
    }

    public byte[] getKeyMaterial(Key key) {
        try {
            return this.doGetKeyMaterial(key);
        }
        catch (Throwable t2) {
            if (t2 instanceof KeyException) {
                throw (KeyException)t2;
            }
            String msg = "Invalid " + this.getId() + " ASN.1 encoding: " + t2.getMessage();
            throw new InvalidKeyException(msg, t2);
        }
    }

    protected byte[] doGetKeyMaterial(Key key) {
        byte unusedBytes;
        byte[] encoded = KeysBridge.getEncoded(key);
        int i = Bytes.indexOf(encoded, this.ASN1_OID);
        Assert.gt(i, -1, "Missing or incorrect algorithm OID.");
        int keyLen = 0;
        if (encoded[i += this.ASN1_OID.length] == 5) {
            unusedBytes = encoded[++i];
            Assert.eq(Integer.valueOf(unusedBytes), 0, "OID NULL terminator should indicate zero unused bytes.");
            ++i;
        }
        if (encoded[i] == 3) {
            int n = ++i;
            keyLen = encoded[n];
            int n2 = ++i;
            ++i;
            unusedBytes = encoded[n2];
            Assert.eq(Integer.valueOf(unusedBytes), 0, "BIT STREAM should not indicate unused bytes.");
            --keyLen;
        } else if (encoded[i] == 4) {
            int n = ++i;
            keyLen = encoded[n];
            if (encoded[++i] == 4) {
                int n3 = ++i;
                ++i;
                keyLen = encoded[n3];
            }
        }
        Assert.eq(keyLen, this.encodedKeyByteLength, "Invalid key length.");
        byte[] result = Arrays.copyOfRange(encoded, i, i + keyLen);
        keyLen = Bytes.length(result);
        Assert.eq(keyLen, this.encodedKeyByteLength, "Invalid key length.");
        return result;
    }

    private void assertLength(byte[] raw, boolean isPublic) {
        int len = Bytes.length(raw);
        if (len != this.encodedKeyByteLength) {
            String msg = "Invalid " + this.getId() + " encoded " + (isPublic ? "PublicKey" : "PrivateKey") + " length. Should be " + Bytes.bytesMsg(this.encodedKeyByteLength) + ", found " + Bytes.bytesMsg(len) + ".";
            throw new InvalidKeyException(msg);
        }
    }

    public PublicKey toPublicKey(byte[] x, Provider provider) {
        this.assertLength(x, true);
        byte[] encoded = Bytes.concat(this.PUBLIC_KEY_ASN1_PREFIX, x);
        X509EncodedKeySpec spec = new X509EncodedKeySpec(encoded);
        JcaTemplate template = new JcaTemplate(this.getJcaName(), provider);
        return template.generatePublic(spec);
    }

    KeySpec privateKeySpec(byte[] d, boolean standard) {
        byte[] prefix = standard ? this.PRIVATE_KEY_ASN1_PREFIX : this.PRIVATE_KEY_JDK11_PREFIX;
        byte[] encoded = Bytes.concat(prefix, d);
        return new PKCS8EncodedKeySpec(encoded);
    }

    public PrivateKey toPrivateKey(byte[] d, Provider provider) {
        this.assertLength(d, false);
        KeySpec spec = this.privateKeySpec(d, true);
        JcaTemplate template = new JcaTemplate(this.getJcaName(), provider);
        return template.generatePrivate(spec);
    }

    public boolean isSignatureCurve() {
        return this.signatureCurve;
    }

    @Override
    public KeyPairBuilder keyPair() {
        return new DefaultKeyPairBuilder(this.getJcaName(), this.keyBitLength);
    }

    public static boolean isEdwards(Key key) {
        if (key == null) {
            return false;
        }
        String alg = Strings.clean(key.getAlgorithm());
        return "EdDSA".equals(alg) || "XDH".equals(alg) || EdwardsCurve.findByKey(key) != null;
    }

    public static PublicKey derivePublic(PrivateKey pk) throws KeyException {
        return EdwardsPublicKeyDeriver.INSTANCE.apply(pk);
    }

    public static EdwardsCurve findById(String id) {
        return REGISTRY.get(id);
    }

    public static EdwardsCurve findByKey(Key key) {
        if (key == null) {
            return null;
        }
        String alg = key.getAlgorithm();
        EdwardsCurve curve = EdwardsCurve.findById(alg);
        if (curve == null) {
            alg = CURVE_NAME_FINDER.apply(key);
            curve = EdwardsCurve.findById(alg);
        }
        byte[] encoded = KeysBridge.findEncoded(key);
        if (curve == null && !Bytes.isEmpty(encoded)) {
            int oidTerminalNode = EdwardsCurve.findOidTerminalNode(encoded);
            curve = BY_OID_TERMINAL_NODE.get(oidTerminalNode);
        }
        if (curve != null && !Bytes.isEmpty(encoded)) {
            try {
                curve.getKeyMaterial(key);
            }
            catch (Throwable ignored) {
                curve = null;
            }
        }
        return curve;
    }

    @Override
    public boolean contains(Key key) {
        EdwardsCurve curve = EdwardsCurve.findByKey(key);
        return curve.equals(this);
    }

    private static int findOidTerminalNode(byte[] encoded) {
        int index = Bytes.indexOf(encoded, ASN1_OID_PREFIX);
        if (index > -1 && (index += ASN1_OID_PREFIX.length) < encoded.length) {
            return encoded[index];
        }
        return -1;
    }

    public static EdwardsCurve forKey(Key key) {
        Assert.notNull(key, "Key cannot be null.");
        EdwardsCurve curve = EdwardsCurve.findByKey(key);
        if (curve == null) {
            String msg = "Unrecognized Edwards Curve key: [" + KeysBridge.toString(key) + "]";
            throw new InvalidKeyException(msg);
        }
        return curve;
    }

    static <K extends Key> K assertEdwards(K key) {
        EdwardsCurve.forKey(key);
        return key;
    }

    static {
        for (EdwardsCurve curve : VALUES) {
            byte subcategoryId = curve.ASN1_OID[curve.ASN1_OID.length - 1];
            BY_OID_TERMINAL_NODE.put(Integer.valueOf(subcategoryId), curve);
            REGISTRY.put(curve.getId(), curve);
            REGISTRY.put(curve.OID, curve);
        }
    }
}

