/*
 * Decompiled with CFR 0.152.
 */
package de.keksuccino.spiffyhud.customization.marker;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonParseException;
import de.keksuccino.spiffyhud.SpiffyHud;
import de.keksuccino.spiffyhud.customization.marker.MarkerData;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Consumer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class MarkerStorage {
    private static final Logger LOGGER = LogManager.getLogger();
    private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
    private static final Path STORAGE_PATH = SpiffyHud.INSTANCE_DIR.toPath().resolve("markers.json");
    private static MarkerStoreData cachedData = new MarkerStoreData();
    private static boolean loaded;

    private MarkerStorage() {
    }

    @NotNull
    public static synchronized List<MarkerData> getMarkers(@NotNull String groupId) {
        String normalized = MarkerStorage.normalizeGroupId(groupId);
        MarkerStorage.ensureLoaded();
        List<MarkerData> stored = cachedData.viewMarkers(normalized);
        if (stored.isEmpty()) {
            return Collections.emptyList();
        }
        ArrayList<MarkerData> copies = new ArrayList<MarkerData>(stored.size());
        for (MarkerData marker : stored) {
            copies.add(marker.copy());
        }
        return Collections.unmodifiableList(copies);
    }

    @NotNull
    public static synchronized List<String> getAllGroupIds() {
        MarkerStorage.ensureLoaded();
        return cachedData.listGroupIds();
    }

    @Nullable
    public static synchronized MarkerData getMarker(@NotNull String groupId, @NotNull String markerName) {
        String normalized = MarkerStorage.normalizeGroupId(groupId);
        MarkerStorage.ensureLoaded();
        MarkerData existing = MarkerStorage.findMarker(cachedData.viewMarkers(normalized), markerName);
        return existing == null ? null : existing.copy();
    }

    public static synchronized boolean addMarker(@NotNull String groupId, @NotNull MarkerData marker) {
        String normalized = MarkerStorage.normalizeGroupId(groupId);
        Objects.requireNonNull(marker, "marker");
        MarkerStorage.ensureLoaded();
        if (!marker.hasValidName()) {
            LOGGER.warn("Ignoring marker without a valid name for group '{}'.", (Object)normalized);
            return false;
        }
        List<MarkerData> group = cachedData.getOrCreateGroup(normalized);
        if (MarkerStorage.findMarker(group, marker.getName()) != null) {
            return false;
        }
        group.add(marker.copy());
        MarkerStorage.persist();
        return true;
    }

    public static synchronized boolean editMarker(@NotNull String groupId, @NotNull String markerName, @NotNull Consumer<MarkerData> editor) {
        String normalized = MarkerStorage.normalizeGroupId(groupId);
        Objects.requireNonNull(editor, "editor");
        MarkerStorage.ensureLoaded();
        List<MarkerData> group = cachedData.viewMarkers(normalized);
        MarkerData existing = MarkerStorage.findMarker(group, markerName);
        if (existing == null) {
            return false;
        }
        MarkerData workingCopy = existing.copy();
        editor.accept(workingCopy);
        if (!workingCopy.hasValidName()) {
            LOGGER.warn("Refusing to apply marker edit for group '{}' because the new name is invalid.", (Object)normalized);
            return false;
        }
        MarkerData conflict = MarkerStorage.findMarker(group, workingCopy.getName());
        if (conflict != null && conflict != existing) {
            return false;
        }
        existing.copyFrom(workingCopy);
        MarkerStorage.persist();
        return true;
    }

    public static synchronized boolean removeMarker(@NotNull String groupId, @NotNull String markerName) {
        String normalized = MarkerStorage.normalizeGroupId(groupId);
        MarkerStorage.ensureLoaded();
        List<MarkerData> group = cachedData.viewMarkers(normalized);
        MarkerData existing = MarkerStorage.findMarker(group, markerName);
        if (existing == null) {
            return false;
        }
        group.remove(existing);
        cachedData.removeGroupIfEmpty(normalized);
        MarkerStorage.persist();
        return true;
    }

    public static synchronized boolean clearGroup(@NotNull String groupId) {
        String normalized = MarkerStorage.normalizeGroupId(groupId);
        MarkerStorage.ensureLoaded();
        boolean dropped = cachedData.dropGroup(normalized);
        if (dropped) {
            MarkerStorage.persist();
        }
        return dropped;
    }

    private static void ensureLoaded() {
        if (loaded) {
            return;
        }
        cachedData = MarkerStorage.readFromDisk();
        loaded = true;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static MarkerStoreData readFromDisk() {
        if (!Files.exists(STORAGE_PATH, new LinkOption[0])) {
            return new MarkerStoreData();
        }
        try (BufferedReader reader = Files.newBufferedReader(STORAGE_PATH, StandardCharsets.UTF_8);){
            MarkerStoreData data = (MarkerStoreData)GSON.fromJson((Reader)reader, MarkerStoreData.class);
            if (data == null) {
                MarkerStoreData markerStoreData2 = new MarkerStoreData();
                return markerStoreData2;
            }
            data.sanitize();
            MarkerStoreData markerStoreData = data;
            return markerStoreData;
        }
        catch (JsonParseException | IOException ex) {
            LOGGER.warn("Failed to read stored markers", ex);
            return new MarkerStoreData();
        }
    }

    private static void persist() {
        try {
            if (STORAGE_PATH.getParent() != null) {
                Files.createDirectories(STORAGE_PATH.getParent(), new FileAttribute[0]);
            }
            try (BufferedWriter writer = Files.newBufferedWriter(STORAGE_PATH, StandardCharsets.UTF_8, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING, StandardOpenOption.WRITE);){
                GSON.toJson((Object)cachedData, (Appendable)writer);
            }
        }
        catch (IOException ex) {
            LOGGER.warn("Failed to persist marker data", (Throwable)ex);
        }
    }

    @Nullable
    private static MarkerData findMarker(@NotNull List<MarkerData> group, @NotNull String name) {
        Objects.requireNonNull(name, "name");
        for (MarkerData marker : group) {
            if (marker == null || !marker.getName().equalsIgnoreCase(name)) continue;
            return marker;
        }
        return null;
    }

    @NotNull
    private static String normalizeGroupId(@NotNull String groupId) {
        Objects.requireNonNull(groupId, "groupId");
        String trimmed = groupId.trim();
        if (trimmed.isEmpty()) {
            throw new IllegalArgumentException("groupId cannot be blank");
        }
        return trimmed;
    }

    private static class MarkerStoreData {
        private Map<String, List<MarkerData>> groups = new HashMap<String, List<MarkerData>>();

        private MarkerStoreData() {
        }

        private void sanitize() {
            if (this.groups == null) {
                this.groups = new HashMap<String, List<MarkerData>>();
                return;
            }
            HashMap<String, List<MarkerData>> sanitized = new HashMap<String, List<MarkerData>>();
            for (Map.Entry<String, List<MarkerData>> entry : this.groups.entrySet()) {
                String normalizedKey;
                String key = entry.getKey();
                if (key == null || (normalizedKey = key.trim()).isEmpty()) continue;
                List<MarkerData> originalList = entry.getValue();
                ArrayList<MarkerData> cleaned = new ArrayList<MarkerData>();
                if (originalList != null) {
                    for (MarkerData marker : originalList) {
                        if (marker == null || !marker.hasValidName()) continue;
                        cleaned.add(marker);
                    }
                }
                sanitized.put(normalizedKey, cleaned);
            }
            this.groups = sanitized;
        }

        private List<MarkerData> viewMarkers(@NotNull String groupId) {
            List<MarkerData> markers = this.groups.get(groupId);
            if (markers == null) {
                return new ArrayList<MarkerData>();
            }
            return markers;
        }

        private List<String> listGroupIds() {
            return new ArrayList<String>(this.groups.keySet());
        }

        private List<MarkerData> getOrCreateGroup(@NotNull String groupId) {
            return this.groups.computeIfAbsent(groupId, ignored -> new ArrayList());
        }

        private void removeGroupIfEmpty(@NotNull String groupId) {
            List<MarkerData> markers = this.groups.get(groupId);
            if (markers != null && markers.isEmpty()) {
                this.groups.remove(groupId);
            }
        }

        private boolean dropGroup(@NotNull String groupId) {
            return this.groups.remove(groupId) != null;
        }
    }
}

