package de.keksuccino.spiffyhud.customization.elements.vanillalike.scoreboard;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.mojang.blaze3d.systems.RenderSystem;
import com.mojang.datafixers.util.Pair;
import de.keksuccino.fancymenu.customization.element.AbstractElement;
import de.keksuccino.fancymenu.customization.element.ElementBuilder;
import de.keksuccino.fancymenu.util.rendering.DrawableColor;
import de.keksuccino.fancymenu.util.rendering.RenderingUtils;
import de.keksuccino.spiffyhud.util.SpiffyAlignment;
import de.keksuccino.fancymenu.util.rendering.gui.GuiGraphics;
import net.minecraft.class_1074;
import net.minecraft.class_124;
import net.minecraft.class_2561;
import net.minecraft.class_266;
import net.minecraft.class_267;
import net.minecraft.class_268;
import net.minecraft.class_269;
import net.minecraft.class_274;
import net.minecraft.class_310;
import net.minecraft.class_327;
import net.minecraft.class_5250;
import net.minecraft.world.scores.*;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import java.util.*;
import java.util.stream.Collectors;

public class VanillaLikeScoreboardElement extends AbstractElement {

    private static final Logger LOGGER = LogManager.getLogger();
    private static final String SPACER = ": ";

    // Minecraft instance for accessing game state
    private final class_310 minecraft = class_310.method_1551();

    // These fields will hold the computed sidebar dimensions and original position
    private int sidebarWidth = 100;
    private int sidebarHeight = 100;
    private int sidebarOriginalX = 0;
    private int sidebarOriginalY = 0;

    // Flag to indicate if the sidebar should actually be drawn
    private boolean renderSidebar = false;

    @NotNull
    public SpiffyAlignment spiffyAlignment = SpiffyAlignment.TOP_LEFT;
    @Nullable
    public DrawableColor customTitleBackgroundColor = null;
    @Nullable
    public DrawableColor customLineBackgroundColor = null;

    public VanillaLikeScoreboardElement(@NotNull ElementBuilder<?, ?> builder) {
        super(builder);
    }

    /**
     * Main render method.
     *
     * First, it calls renderScoreboard (with no offset) to update the sidebar’s dimensions and original position.
     * Then, it calculates an aligned position based on the element’s absolute bounds and renders the scoreboard
     * with an offset so that it appears inside the element.
     */
    @Override
    public void render(@NotNull GuiGraphics graphics, int mouseX, int mouseY, float partial) {

        if (this.minecraft.field_1724 == null || this.minecraft.field_1687 == null) {
            return;
        }

        // --- Update sidebar dimensions (without applying any offset) ---
        // This call sets sidebarWidth, sidebarHeight, sidebarOriginalX, and sidebarOriginalY.
        this.renderSidebar = false;
        this.renderScoreboard(graphics, 0, 0, false);

        // Calculate aligned position based on the element's absolute bounds
        int elementX = this.getAbsoluteX();
        int elementY = this.getAbsoluteY();
        Integer[] alignedPosition = SpiffyAlignment.calculateElementBodyPosition(
                this.spiffyAlignment,
                elementX,
                elementY,
                this.getAbsoluteWidth(),
                this.getAbsoluteHeight(),
                this.sidebarWidth,
                this.sidebarHeight
        );

        // Compute the offset required so that the sidebar renders at the correct aligned position.
        int offsetX = alignedPosition[0] - this.sidebarOriginalX;
        int offsetY = alignedPosition[1] - this.sidebarOriginalY;

        RenderSystem.enableBlend();

        // Render the scoreboard applying the offset to all drawing coordinates.
        this.renderSidebar = true;
        this.renderScoreboard(graphics, offsetX, offsetY, true);

        RenderingUtils.resetShaderColor(graphics);

    }

    /**
     * Renders the scoreboard.
     *
     * @param graphics    The graphics context.
     * @param offsetX     Horizontal offset to add to drawing coordinates.
     * @param offsetY     Vertical offset to add to drawing coordinates.
     * @param applyOffset If true, the computed offset is applied while drawing.
     */
    private void renderScoreboard(GuiGraphics graphics, int offsetX, int offsetY, boolean applyOffset) {

        // Enable blending and set the shader color with the desired opacity.
        RenderSystem.enableBlend();
        graphics.setColor(1.0f, 1.0f, 1.0f, this.opacity);

        class_269 scoreboard = this.minecraft.field_1687.method_8428();
        class_266 objective = null;
        class_268 playerTeam = scoreboard.method_1164(this.minecraft.field_1724.method_5820());
        int colorId = -1;
        if (playerTeam != null) {
            colorId = playerTeam.method_1202().method_536();
        }
        if (playerTeam != null && colorId >= 0) {
            objective = scoreboard.method_1189(3 + colorId);
        }
        // Fallback to display objective 1 if necessary.
        class_266 objectiveToRender = (objective != null) ? objective : scoreboard.method_1189(1);
        // In editor mode, use a dummy objective.
        if (isEditor()) {
            objectiveToRender = new class_266(scoreboard, "", class_274.field_1468, class_2561.method_43473(), class_274.class_275.field_1472);
        }
        if (objectiveToRender != null) {
            displayScoreboardSidebar(graphics, objectiveToRender, offsetX, offsetY, applyOffset);
        }

        graphics.setColor(1.0f, 1.0f, 1.0f, 1.0f);

    }

    /**
     * Renders the scoreboard sidebar.
     *
     * All drawing coordinates (for background, score lines, and title) are adjusted by (offsetX, offsetY)
     * if applyOffset is true. This ensures that the sidebar is drawn inside the element’s bounds.
     *
     * @param guiGraphics The graphics context.
     * @param objective   The scoreboard objective to render.
     * @param offsetX     Horizontal offset to apply.
     * @param offsetY     Vertical offset to apply.
     * @param applyOffset If true, the offset is added to all drawing coordinates.
     */
    private void displayScoreboardSidebar(GuiGraphics guiGraphics, class_266 objective, int offsetX, int offsetY, boolean applyOffset) {

        class_269 scoreboard = objective.method_1117();

        // Filter out invalid scores and limit to 15 entries
        Collection<class_267> rawScores = scoreboard.method_1184(objective);
        List<class_267> validScores = rawScores.stream()
                .filter(score -> (score.method_1129() != null) && !score.method_1129().startsWith("#"))
                .collect(Collectors.toList());
        List<class_267> scoresToDisplay = validScores.size() > 15
                ? Lists.newArrayList(Iterables.skip(validScores, validScores.size() - 15))
                : validScores;

        // Build a list pairing each score with its formatted display name
        ArrayList<Pair<class_267, class_5250>> scoreComponents = Lists.newArrayListWithCapacity(scoresToDisplay.size());
        class_2561 title = objective.method_1114();

        // In editor mode, use dummy values
        if (isEditor()) {
            title = class_2561.method_43471("spiffyhud.elements.dummy.scoreboard_sidebar.title").method_27692(class_124.field_1067);
            List<class_267> dummyEntries = new ArrayList<>();
            String dummyLineText = class_1074.method_4662("spiffyhud.elements.dummy.scoreboard_sidebar.line");
            for (int i = 0; i < 6; i++) {
                dummyEntries.add(new DummyScore(objective, dummyLineText, 0));
            }
            dummyEntries.add(new DummyScore(objective, "", 0));
            scoresToDisplay = dummyEntries;
        }

        // Determine the maximum width required by the title or any score entry.
        int titleWidth = this.getFont().method_27525(title);
        int spacerWidth = this.getFont().method_1727(SPACER);
        int maxEntryWidth = titleWidth;
        for (class_267 scoreEntry : scoresToDisplay) {
            class_268 team = scoreboard.method_1164(scoreEntry.method_1129());
            class_5250 scoreComponent = class_268.method_1142(team, class_2561.method_43470(scoreEntry.method_1129()));
            scoreComponents.add(Pair.of(scoreEntry, scoreComponent));
            int entryWidth = this.getFont().method_27525(scoreComponent) + spacerWidth + this.getFont().method_1727(Integer.toString(scoreEntry.method_1126()));
            maxEntryWidth = Math.max(maxEntryWidth, entryWidth);
        }

        // Compute base positions using screen dimensions.
        int numberOfLines = scoresToDisplay.size();
        int totalLineHeight = numberOfLines * this.getFont().field_2000;
        int baseY = getScreenHeight() / 2 + totalLineHeight / 3;
        int baseX = getScreenWidth() - maxEntryWidth - 3;

        // Background colors (custom if set)
        int lineBackgroundColor = this.minecraft.field_1690.method_19345(0.3f);
        if (this.customLineBackgroundColor != null) {
            lineBackgroundColor = this.customLineBackgroundColor.getColorInt();
        }
        int titleBackgroundColor = this.minecraft.field_1690.method_19345(0.4f);
        if (this.customTitleBackgroundColor != null) {
            titleBackgroundColor = this.customTitleBackgroundColor.getColorInt();
        }

        // Compute the top Y position of the sidebar
        int topY = baseY - numberOfLines * this.getFont().field_2000;
        // Record sidebar dimensions and original position (before applying any offset)
        this.sidebarWidth = Math.max(1, maxEntryWidth + 4); // adding padding
        this.sidebarHeight = Math.max(1, baseY - (topY - this.getFont().field_2000 - 1));
        this.sidebarOriginalX = baseX - 2;
        this.sidebarOriginalY = topY - this.getFont().field_2000 - 1;

        // Apply offset if needed
        int effectiveBaseX = baseX + (applyOffset ? offsetX : 0);
        int effectiveBaseY = baseY + (applyOffset ? offsetY : 0);

        // If we are in rendering mode, draw each score line and then the title background.
        if (this.renderSidebar) {
            int lineIndex = 0;
            for (Pair<class_267, class_5250> scorePair : scoreComponents) {
                lineIndex++;
                class_267 currentScore = scorePair.getFirst();
                class_2561 entryComponent = scorePair.getSecond();
                String scoreText = "" + class_124.field_1061 + currentScore.method_1126();
                // Compute positions for this line
                int lineX = effectiveBaseX;
                int lineY = effectiveBaseY - lineIndex * 9;
                // rightX is computed from the screen width and offset as well
                int rightX = (getScreenWidth() - 3 + 2) + (applyOffset ? offsetX : 0);
                // Draw background for this score line
                guiGraphics.fill(lineX - 2, lineY, rightX, lineY + 9, lineBackgroundColor);
                // Draw the player's name and score
                guiGraphics.drawString(this.getFont(), entryComponent, lineX, lineY, -1, false);
                guiGraphics.drawString(this.getFont(), scoreText, rightX - this.getFont().method_1727(scoreText), lineY, -1, false);
                // On the last line, also draw the title background and title text
                if (lineIndex == scoresToDisplay.size()) {
                    guiGraphics.fill(lineX - 2, lineY - 9 - 1, rightX, lineY - 1, titleBackgroundColor);
                    guiGraphics.fill(lineX - 2, lineY - 1, rightX, lineY, lineBackgroundColor);
                    class_327 font = this.getFont();
                    int titleX = lineX + maxEntryWidth / 2 - titleWidth / 2;
                    guiGraphics.drawString(font, title, titleX, lineY - 9, -1, false);
                }
            }
        }

    }

    /**
     * Returns the current Font instance.
     */
    private class_327 getFont() {
        return class_310.method_1551().field_1772;
    }

    /**
     * The absolute width of this element equals the sidebar width.
     */
    @Override
    public int getAbsoluteWidth() {
        return this.sidebarWidth;
    }

    /**
     * The absolute height of this element equals the sidebar height.
     */
    @Override
    public int getAbsoluteHeight() {
        return this.sidebarHeight;
    }

    /**
     * DummyScore is used in editor mode to simulate scoreboard entries.
     */
    private static class DummyScore extends class_267 {
        int score;

        public DummyScore(class_266 objective, String display, int score) {
            super(objective.method_1117(), objective, display);
            this.score = score;
        }

        @Override
        public int method_1126() {
            return this.score;
        }
    }

}
