package de.keksuccino.spiffyhud.customization.elements.vanillalike.scoreboard;

import com.google.common.collect.Lists;
import com.mojang.blaze3d.systems.RenderSystem;
import de.keksuccino.fancymenu.customization.element.AbstractElement;
import de.keksuccino.fancymenu.customization.element.ElementBuilder;
import de.keksuccino.fancymenu.util.rendering.DrawableColor;
import de.keksuccino.fancymenu.util.rendering.RenderingUtils;
import de.keksuccino.spiffyhud.util.SpiffyAlignment;
import net.minecraft.class_1074;
import net.minecraft.class_124;
import net.minecraft.class_2561;
import net.minecraft.class_266;
import net.minecraft.class_268;
import net.minecraft.class_269;
import net.minecraft.class_274;
import net.minecraft.class_310;
import net.minecraft.class_327;
import net.minecraft.class_332;
import net.minecraft.class_8646;
import net.minecraft.class_9011;
import net.minecraft.class_9025;
import net.minecraft.world.scores.*;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import java.util.*;
import java.util.Comparator;

public class VanillaLikeScoreboardElement extends AbstractElement {

    private static final Logger LOGGER = LogManager.getLogger();
    private static final String SPACER = ": ";
    private static final Comparator<class_9011> SCORE_DISPLAY_ORDER = Comparator.comparing(class_9011::comp_2128)
            .reversed()
            .thenComparing(class_9011::comp_2127, String.CASE_INSENSITIVE_ORDER);

    // Minecraft instance for accessing game state
    private final class_310 minecraft = class_310.method_1551();

    // These fields will hold the computed sidebar dimensions and original position
    private int sidebarWidth = 100;
    private int sidebarHeight = 100;
    private int sidebarOriginalX = 0;
    private int sidebarOriginalY = 0;

    // Flag to indicate if the sidebar should actually be drawn
    private boolean renderSidebar = false;

    @NotNull
    public SpiffyAlignment spiffyAlignment = SpiffyAlignment.TOP_LEFT;
    @Nullable
    public DrawableColor customTitleBackgroundColor = null;
    @Nullable
    public DrawableColor customLineBackgroundColor = null;

    public VanillaLikeScoreboardElement(@NotNull ElementBuilder<?, ?> builder) {
        super(builder);
    }

    /**
     * Main render method.
     *
     * First, it calls renderScoreboard (with no offset) to update the sidebar's dimensions and original position.
     * Then, it calculates an aligned position based on the element's absolute bounds and renders the scoreboard
     * with an offset so that it appears inside the element.
     */
    @Override
    public void method_25394(@NotNull class_332 graphics, int mouseX, int mouseY, float partial) {

        if (this.minecraft.field_1724 == null || this.minecraft.field_1687 == null) {
            return;
        }

        // --- Update sidebar dimensions (without applying any offset) ---
        // This call sets sidebarWidth, sidebarHeight, sidebarOriginalX, and sidebarOriginalY.
        this.renderSidebar = false;
        this.renderScoreboard(graphics, 0, 0, false);

        // Calculate aligned position based on the element's absolute bounds
        int elementX = this.getAbsoluteX();
        int elementY = this.getAbsoluteY();
        Integer[] alignedPosition = SpiffyAlignment.calculateElementBodyPosition(
                this.spiffyAlignment,
                elementX,
                elementY,
                this.getAbsoluteWidth(),
                this.getAbsoluteHeight(),
                this.sidebarWidth,
                this.sidebarHeight
        );

        // Compute the offset required so that the sidebar renders at the correct aligned position.
        int offsetX = alignedPosition[0] - this.sidebarOriginalX;
        int offsetY = alignedPosition[1] - this.sidebarOriginalY;

        RenderSystem.enableBlend();

        // Render the scoreboard applying the offset to all drawing coordinates.
        this.renderSidebar = true;
        this.renderScoreboard(graphics, offsetX, offsetY, true);

        RenderingUtils.resetShaderColor(graphics);

    }

    /**
     * Renders the scoreboard.
     *
     * @param graphics    The graphics context.
     * @param offsetX     Horizontal offset to add to drawing coordinates.
     * @param offsetY     Vertical offset to add to drawing coordinates.
     * @param applyOffset If true, the computed offset is applied while drawing.
     */
    private void renderScoreboard(class_332 graphics, int offsetX, int offsetY, boolean applyOffset) {

        // Enable blending and set the shader color with the desired opacity.
        RenderSystem.enableBlend();
        graphics.method_51422(1.0f, 1.0f, 1.0f, this.opacity);

        class_269 scoreboard = this.minecraft.field_1687.method_8428();
        class_266 objective = null;
        class_268 playerTeam = scoreboard.method_1164(this.minecraft.field_1724.method_5820());
        
        if (playerTeam != null) {
            class_8646 displaySlot = class_8646.method_52622(playerTeam.method_1202());
            if (displaySlot != null) {
                objective = scoreboard.method_1189(displaySlot);
            }
        }
        
        // Fallback to SIDEBAR display objective if necessary
        class_266 objectiveToRender = (objective != null) ? objective : scoreboard.method_1189(class_8646.field_45157);
        
        // In editor mode, use a dummy objective
        if (isEditor()) {
            objectiveToRender = new DummyObjective(scoreboard);
        }
        
        if (objectiveToRender != null) {
            displayScoreboardSidebar(graphics, objectiveToRender, offsetX, offsetY, applyOffset);
        }

        graphics.method_51422(1.0f, 1.0f, 1.0f, 1.0f);
    }

    /**
     * Record class to hold display entry information
     */
    private record DisplayEntry(class_2561 name, class_2561 score, int scoreWidth) {}

    /**
     * Renders the scoreboard sidebar.
     *
     * All drawing coordinates (for background, score lines, and title) are adjusted by (offsetX, offsetY)
     * if applyOffset is true. This ensures that the sidebar is drawn inside the element's bounds.
     *
     * @param guiGraphics The graphics context.
     * @param objective   The scoreboard objective to render.
     * @param offsetX     Horizontal offset to apply.
     * @param offsetY     Vertical offset to apply.
     * @param applyOffset If true, the offset is added to all drawing coordinates.
     */
    private void displayScoreboardSidebar(class_332 guiGraphics, class_266 objective, int offsetX, int offsetY, boolean applyOffset) {
        class_269 scoreboard = objective.method_1117();
        
        // List to hold display entries
        DisplayEntry[] displayEntries;
        class_2561 title = objective.method_1114();
        
        // In editor mode, use dummy values
        if (isEditor()) {
            title = class_2561.method_43471("spiffyhud.elements.dummy.scoreboard_sidebar.title").method_27692(class_124.field_1067);
            String dummyLineText = class_1074.method_4662("spiffyhud.elements.dummy.scoreboard_sidebar.line");
            displayEntries = new DisplayEntry[6];
            for (int i = 0; i < 6; i++) {
                class_2561 name = class_2561.method_43470(dummyLineText);
                class_2561 score = class_2561.method_43470("" + i);
                displayEntries[i] = new DisplayEntry(name, score, this.getFont().method_27525(score));
            }
        } else {
            // Get player scores using the 1.21.1 API
            displayEntries = scoreboard.method_1184(objective)
                .stream()
                .filter(entry -> !entry.method_55385())
                .sorted(SCORE_DISPLAY_ORDER)
                .limit(15)
                .map(entry -> {
                    class_268 team = scoreboard.method_1164(entry.comp_2127());
                    class_2561 name = entry.method_55387();
                    class_2561 formattedName = class_268.method_1142(team, name);
                    class_2561 scoreText = entry.method_55386(objective.method_55380(class_9025.field_47567));
                    int scoreWidth = this.getFont().method_27525(scoreText);
                    return new DisplayEntry(formattedName, scoreText, scoreWidth);
                })
                .toArray(DisplayEntry[]::new);
        }

        // Determine the maximum width required by the title or any score entry.
        int titleWidth = this.getFont().method_27525(title);
        int maxEntryWidth = titleWidth;
        int spacerWidth = this.getFont().method_1727(SPACER);

        for (DisplayEntry entry : displayEntries) {
            int entryWidth = this.getFont().method_27525(entry.name) + (entry.scoreWidth > 0 ? spacerWidth + entry.scoreWidth : 0);
            maxEntryWidth = Math.max(maxEntryWidth, entryWidth);
        }

        // Compute base positions using screen dimensions.
        int numberOfLines = displayEntries.length;
        int lineHeight = 9; // Standard line height for scoreboard
        int totalLineHeight = numberOfLines * lineHeight;
        int baseY = getScreenHeight() / 2 + totalLineHeight / 3;
        int baseX = getScreenWidth() - maxEntryWidth - 3;

        // Background colors (custom if set)
        int lineBackgroundColor = this.minecraft.field_1690.method_19345(0.3f);
        if (this.customLineBackgroundColor != null) {
            lineBackgroundColor = this.customLineBackgroundColor.getColorInt();
        }
        int titleBackgroundColor = this.minecraft.field_1690.method_19345(0.4f);
        if (this.customTitleBackgroundColor != null) {
            titleBackgroundColor = this.customTitleBackgroundColor.getColorInt();
        }

        // Compute the top Y position of the sidebar
        int topY = baseY - numberOfLines * lineHeight;
        // Record sidebar dimensions and original position (before applying any offset)
        this.sidebarWidth = Math.max(1, maxEntryWidth + 4); // adding padding
        this.sidebarHeight = Math.max(1, baseY - (topY - lineHeight - 1));
        this.sidebarOriginalX = baseX - 2;
        this.sidebarOriginalY = topY - lineHeight - 1;

        // Apply offset if needed
        int effectiveBaseX = baseX + (applyOffset ? offsetX : 0);
        int effectiveBaseY = baseY + (applyOffset ? offsetY : 0);

        // If we are in rendering mode, draw each score line and then the title background.
        if (this.renderSidebar) {
            guiGraphics.method_51448().method_22903();
            
            // Draw managed to ensure blending and other state is properly handled
            int finalLineBackgroundColor = lineBackgroundColor;
            int finalTitleBackgroundColor = titleBackgroundColor;
            int finalMaxEntryWidth = maxEntryWidth;
            class_2561 finalTitle = title;
            guiGraphics.method_51741(() -> {
                int rightX = (getScreenWidth() - 3 + 2) + (applyOffset ? offsetX : 0);
                
                for (int i = 0; i < displayEntries.length; i++) {
                    DisplayEntry entry = displayEntries[i];
                    int lineY = effectiveBaseY - (i + 1) * lineHeight;
                    
                    // Draw background for this score line
                    guiGraphics.method_25294(effectiveBaseX - 2, lineY, rightX, lineY + lineHeight, finalLineBackgroundColor);
                    
                    // Draw the player's name and score
                    guiGraphics.method_51439(this.getFont(), entry.name, effectiveBaseX, lineY, -1, false);
                    guiGraphics.method_51439(this.getFont(), entry.score, rightX - entry.scoreWidth, lineY, -1, false);
                    
                    // On the last line, also draw the title background and title text
                    if (i == displayEntries.length - 1) {
                        guiGraphics.method_25294(effectiveBaseX - 2, lineY - lineHeight - 1, rightX, lineY - 1, finalTitleBackgroundColor);
                        guiGraphics.method_25294(effectiveBaseX - 2, lineY - 1, rightX, lineY, finalLineBackgroundColor);
                        class_327 font = this.getFont();
                        int titleX = effectiveBaseX + finalMaxEntryWidth / 2 - titleWidth / 2;
                        guiGraphics.method_51439(font, finalTitle, titleX, lineY - lineHeight, -1, false);
                    }
                }
            });
            
            guiGraphics.method_51448().method_22909();
        }
    }

    /**
     * Returns the current Font instance.
     */
    private class_327 getFont() {
        return class_310.method_1551().field_1772;
    }

    /**
     * The absolute width of this element equals the sidebar width.
     */
    @Override
    public int getAbsoluteWidth() {
        return this.sidebarWidth;
    }

    /**
     * The absolute height of this element equals the sidebar height.
     */
    @Override
    public int getAbsoluteHeight() {
        return this.sidebarHeight;
    }

    /**
     * DummyObjective is used in editor mode to simulate a scoreboard objective.
     */
    private static class DummyObjective extends class_266 {
        public DummyObjective(class_269 scoreboard) {
            // Constructor parameters in 1.21.1:
            // scoreboard, name, criteria, displayName, renderType, displayAutoUpdate, numberFormat
            super(
                scoreboard, 
                "dummy", 
                class_274.field_1468,
                class_2561.method_43470("Scoreboard"), 
                class_274.class_275.field_1472,
                false, 
                null
            );
        }
    }
}
