package de.keksuccino.spiffyhud.util.level;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import net.minecraft.class_2338;
import net.minecraft.class_2378;
import net.minecraft.class_2960;
import net.minecraft.class_3195;
import net.minecraft.class_3218;
import net.minecraft.class_5321;
import net.minecraft.class_5455;
import net.minecraft.class_7924;
import org.jetbrains.annotations.NotNull;

/**
 * Utility class for structure-related operations.
 */
public class StructureUtils {

    /**
     * Checks if a BlockPos is within a specific structure.
     *
     * @param level The server level to check in
     * @param pos The position to check
     * @param structure The structure key to check for
     * @return true if the position is within the structure, false otherwise
     */
    public static boolean isInStructure(@NotNull class_3218 level, @NotNull class_2338 pos, @NotNull class_5321<class_3195> structure) {
        if (!level.method_8477(pos)) {
            return false; // Position not loaded, can't check
        }
        List<class_5321<class_3195>> structures = getAllStructuresAt(level, pos);
        for (class_5321<class_3195> key : structures) {
            if (key.toString().equals(structure.toString())) return true;
        }
        return false;
    }

    /**
     * Gets all structures at a specific BlockPos.
     *
     * @param level The server level to check in
     * @param pos The position to check
     * @return A list of ResourceKeys for all structures at this position
     */
    @NotNull
    public static List<class_5321<class_3195>> getAllStructuresAt(@NotNull class_3218 level, @NotNull class_2338 pos) {

        if (!level.method_8477(pos)) {
            return List.of(); // Position not loaded, can't check
        }

        // Get all structures in the registry
        class_2378<class_3195> structureRegistry = level.method_30349().method_30530(class_7924.field_41246);

        List<class_5321<class_3195>> keys = new ArrayList<>();
        level.method_27056().method_41037(pos).forEach((structure, longs) -> {
            var structureKey = structureRegistry.method_29113(structure);
            structureKey.ifPresent(keys::add);
        });
        return keys;

    }

    /**
     * Gets a structure resource key from a string identifier.
     *
     * @param structureId The structure identifier (e.g., "minecraft:mansion")
     * @return The ResourceKey for the structure
     */
    @NotNull
    public static class_5321<class_3195> getStructureKey(@NotNull String structureId) {
        class_2960 resourceLocation = class_2960.method_60654(structureId);
        return getStructureKey(resourceLocation);
    }

    /**
     * Gets a structure resource key from a ResourceLocation.
     *
     * @param location The ResourceLocation for the structure
     * @return The ResourceKey for the structure
     */
    @NotNull
    public static class_5321<class_3195> getStructureKey(@NotNull class_2960 location) {
        return class_5321.method_29179(class_7924.field_41246, location);
    }

    /**
     * Gets all available structure resource keys from the registry.
     *
     * @param registryAccess The registry access to get structures from
     * @return A list of all structure resource keys
     */
    @NotNull
    public static List<class_5321<class_3195>> getAllStructureKeys(@NotNull class_5455 registryAccess) {
        class_2378<class_3195> structureRegistry = registryAccess.method_30530(class_7924.field_41246);
        return new ArrayList<>(structureRegistry.method_42021());
    }

    /**
     * Tries to find a structure key by name, returning an Optional result.
     *
     * @param registryAccess The registry access to search in
     * @param structureName The name of the structure to find
     * @return An Optional containing the structure key if found, or empty if not found
     */
    @NotNull
    public static Optional<class_5321<class_3195>> findStructureKey(@NotNull class_5455 registryAccess, @NotNull String structureName) {
        try {
            class_2960 resourceLocation = class_2960.method_60654(structureName);
            class_5321<class_3195> key = class_5321.method_29179(class_7924.field_41246, resourceLocation);
            // Verify the key exists in the registry
            class_2378<class_3195> structureRegistry = registryAccess.method_30530(class_7924.field_41246);
            if (structureRegistry.method_35842(key)) {
                return Optional.of(key);
            }
            return Optional.empty();
        } catch (Exception e) {
            return Optional.empty();
        }
    }

    @NotNull
    public static List<String> convertStructureKeysToStrings(@NotNull List<class_5321<class_3195>> keys) {
        List<String> stringKeys = new ArrayList<>();
        keys.forEach(structureResourceKey -> stringKeys.add(structureResourceKey.method_29177().toString()));
        return stringKeys;
    }

}