package de.keksuccino.spiffyhud.util.rendering;

import com.mojang.blaze3d.systems.RenderSystem;
import de.keksuccino.fancymenu.util.rendering.gui.GuiGraphics;
import de.keksuccino.fancymenu.util.rendering.gui.GuiMatrix4f;
import net.minecraft.class_1158;
import net.minecraft.class_1160;
import net.minecraft.class_1309;
import net.minecraft.class_308;
import net.minecraft.class_310;
import net.minecraft.class_3532;
import net.minecraft.class_4048;
import net.minecraft.class_4050;
import net.minecraft.class_898;

public class GuiEntityRenderer {

    private static final float DEFAULT_FACING = 180.0f; // Default inventory facing
    private static final float MAX_BODY_DELTA = 30.0f;    // Body rotation cap relative to default
    private static final float MAX_HEAD_DELTA = 30.0f;    // Head rotation cap relative to rendered body

    // Instance fields to track rendered rotations.
    // For body rotation:
    private float lastOriginalBodyRotation = Float.NaN;
    private float renderedBodyRotation = DEFAULT_FACING;
    // For head rotation:
    private float lastOriginalHeadRotation = Float.NaN;
    private float renderedHeadRotation = DEFAULT_FACING;

    /**
     * Renders a living entity into a GUI box.
     * <p>
     * The entity is scaled using its bounding dimensions from Pose.STANDING so that its bounding box fits
     * entirely within the target box defined by (posX, posY, boxWidth, boxHeight) and is centered.
     * <p>
     * The rendered body rotation is updated by applying only the incremental difference (delta)
     * from the previous original body rotation value, then clamped to lie within ±30° of DEFAULT_FACING.
     * The head rotation is similarly updated based on the incremental change, but then clamped to within ±30°
     * relative to the rendered body rotation. This prevents either from drifting too far, even if the original
     * rotations continue to increase.
     * After rendering, the entity's original rotations are restored.
     *
     * @param graphics  the GUI graphics context used for rendering
     * @param posX      the x-coordinate of the top-left corner of the target box
     * @param posY      the y-coordinate of the top-left corner of the target box
     * @param boxWidth  the width of the target box (in pixels)
     * @param boxHeight the height of the target box (in pixels)
     * @param opacity   the opacity of the rendered entity (0.0 = fully transparent, 1.0 = fully opaque)
     * @param entity    the LivingEntity to render
     */
    public void renderEntity(GuiGraphics graphics, int posX, int posY, int boxWidth, int boxHeight, float opacity, class_1309 entity) {

        // Obtain the entity's bounding dimensions using Pose.STANDING.
        class_4048 dimensions = entity.method_18377(class_4050.field_18076);
        // Compute scale factors so that the entity's bounding box fits entirely within the target box.
        float scaleFromWidth = (float) boxWidth / dimensions.field_18067;
        float scaleFromHeight = (float) boxHeight / dimensions.field_18068;
        float uniformScale = Math.min(scaleFromWidth, scaleFromHeight);

        // Save original rotation values for later restoration.
        float origYBodyRot = entity.field_6283;
        float origYRot = entity.method_36454();
        float origXRot = entity.method_36455();
        float origYHeadRot = entity.field_6241;
        float origYHeadRotO = entity.field_6259;

        // ----- Update Body Rotation Using Delta and Clamp -----
        float currentOriginalBody = origYBodyRot;
        if (Float.isNaN(lastOriginalBodyRotation)) {
            lastOriginalBodyRotation = currentOriginalBody;
            renderedBodyRotation = currentOriginalBody;
        }
        // Compute the incremental difference (wrapped properly).
        float bodyDelta = class_3532.method_15393(currentOriginalBody - lastOriginalBodyRotation);
        renderedBodyRotation += bodyDelta;
        // Clamp the rendered body rotation to within DEFAULT_FACING ± MAX_BODY_DELTA.
        renderedBodyRotation = class_3532.method_15363(renderedBodyRotation, DEFAULT_FACING - MAX_BODY_DELTA, DEFAULT_FACING + MAX_BODY_DELTA);
        lastOriginalBodyRotation = currentOriginalBody;
        // Override the entity's body rotation for rendering.
        entity.field_6283 = renderedBodyRotation;

        // ----- Update Head Rotation Using Delta and Clamp Relative to Body -----
        float currentOriginalHead = origYHeadRot;
        if (Float.isNaN(lastOriginalHeadRotation)) {
            lastOriginalHeadRotation = currentOriginalHead;
            renderedHeadRotation = currentOriginalHead;
        }
        float headDelta = class_3532.method_15393(currentOriginalHead - lastOriginalHeadRotation);
        float newRenderedHead = renderedHeadRotation + headDelta;
        // Clamp the head rotation so it stays within ±MAX_HEAD_DELTA of the rendered body rotation.
        newRenderedHead = class_3532.method_15363(newRenderedHead, renderedBodyRotation - MAX_HEAD_DELTA, renderedBodyRotation + MAX_HEAD_DELTA);
        lastOriginalHeadRotation = currentOriginalHead;
        renderedHeadRotation = newRenderedHead;
        // Override the entity's head rotation.
        entity.field_6241 = newRenderedHead;
        entity.field_6259 = newRenderedHead;
        entity.method_36456(newRenderedHead);

        // ----- Transformation and Rendering -----
        graphics.pose().method_22903();
        // Translate to the center of the target box.
        graphics.pose().method_22904(posX + boxWidth / 2.0, posY + boxHeight / 2.0, 50.0);
        // Apply uniform scaling (note: negative on Z to mirror correctly).
        graphics.pose().method_34425(new GuiMatrix4f().scaling(uniformScale, uniformScale, -uniformScale));
        // Apply a base rotation: rotate 180° about the Z-axis so the entity faces the viewer.
        class_1158 baseRotation = new class_1158(new class_1160(0.0f, 0.0f, 1.0f), (float) Math.PI, false);
        graphics.pose().method_22907(baseRotation);
        // Shift upward so that the entity's bounding box is vertically centered.
        // Since the pivot is at the feet, translate upward by half of the entity's height.
        graphics.pose().method_22904(0.0, -(dimensions.field_18068 / 2.0f), 0.0);

        // Setup lighting for GUI entity rendering.
        class_308.method_34742();
        // Enable blending and set the shader color with the desired opacity.
        RenderSystem.enableBlend();
        graphics.setColor(1.0f, 1.0f, 1.0f, opacity);

        // Retrieve the entity render dispatcher and disable shadow rendering.
        class_898 dispatcher = class_310.method_1551().method_1561();
        dispatcher.method_3948(false);
        RenderSystem.runAsFancy(() -> {
            dispatcher.method_3954(entity, 0.0, 0.0, 0.0, 0.0f, 1.0f,
                    graphics.pose(), graphics.bufferSource(), 0xF000F0);
        });
        graphics.flush();
        dispatcher.method_3948(true);
        graphics.pose().method_22909();
        class_308.method_24211();
        graphics.setColor(1.0f, 1.0f, 1.0f, 1.0f);

        // Restore the entity's original rotation values.
        entity.field_6283 = origYBodyRot;
        entity.method_36456(origYRot);
        entity.method_36457(origXRot);
        entity.field_6241 = origYHeadRot;
        entity.field_6259 = origYHeadRotO;

    }

}
