package de.keksuccino.spiffyhud.customization.elements.vanillalike.contextualbar;

import de.keksuccino.fancymenu.customization.element.AbstractElement;
import de.keksuccino.fancymenu.customization.element.ElementBuilder;
import de.keksuccino.spiffyhud.util.rendering.SpiffyRenderUtils;
import net.minecraft.class_10799;
import net.minecraft.class_11200;
import net.minecraft.class_11208;
import net.minecraft.class_11326;
import net.minecraft.class_1316;
import net.minecraft.class_1937;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_327;
import net.minecraft.class_332;
import net.minecraft.class_3532;
import net.minecraft.class_746;
import net.minecraft.class_9848;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jetbrains.annotations.NotNull;

public class VanillaLikeContextualBarElement extends AbstractElement {

    private static final Logger LOGGER = LogManager.getLogger();

    // Experience bar sprites
    private static final class_2960 EXPERIENCE_BAR_BACKGROUND_SPRITE = class_2960.method_60656("hud/experience_bar_background");
    private static final class_2960 EXPERIENCE_BAR_PROGRESS_SPRITE = class_2960.method_60656("hud/experience_bar_progress");

    // Jump bar sprites
    private static final class_2960 JUMP_BAR_BACKGROUND_SPRITE = class_2960.method_60656("hud/jump_bar_background");
    private static final class_2960 JUMP_BAR_COOLDOWN_SPRITE = class_2960.method_60656("hud/jump_bar_cooldown");
    private static final class_2960 JUMP_BAR_PROGRESS_SPRITE = class_2960.method_60656("hud/jump_bar_progress");

    // Locator bar sprites
    private static final class_2960 LOCATOR_BAR_BACKGROUND = class_2960.method_60656("hud/locator_bar_background");
    private static final class_2960 LOCATOR_BAR_ARROW_UP = class_2960.method_60656("hud/locator_bar_arrow_up");
    private static final class_2960 LOCATOR_BAR_ARROW_DOWN = class_2960.method_60656("hud/locator_bar_arrow_down");

    private static final int BAR_WIDTH = 182;
    private static final int BAR_HEIGHT = 5;
    private static final int EXPERIENCE_BAR_DISPLAY_TICKS = 100;

    // Locator bar constants
    private static final int DOT_SIZE = 9;
    private static final int VISIBLE_DEGREE_RANGE = 60;
    private static final int ARROW_WIDTH = 7;
    private static final int ARROW_HEIGHT = 5;

    private final class_310 minecraft = class_310.method_1551();
    
    // Property to force showing locator bar
    public boolean alwaysShowLocatorBar = false;
    
    // Property to always show experience bar
    public boolean alwaysShowExperienceBar = false;

    // Enum to track which contextual bar to show
    private enum ContextualInfo {
        EMPTY,
        EXPERIENCE,
        LOCATOR,
        JUMPABLE_VEHICLE
    }

    public VanillaLikeContextualBarElement(@NotNull ElementBuilder<?, ?> builder) {
        super(builder);
    }

    /**
     * Renders the appropriate contextual bar within this element's bounds.
     *
     * @param graphics The graphics context.
     * @param mouseX   The current mouse X-coordinate.
     * @param mouseY   The current mouse Y-coordinate.
     * @param partial  Partial ticks.
     */
    @Override
    public void method_25394(@NotNull class_332 graphics, int mouseX, int mouseY, float partial) {
        // Do nothing if the player or level is missing.
        if (this.minecraft.field_1724 == null || this.minecraft.field_1687 == null) {
            return;
        }

        // Get the absolute position and dimensions of this element.
        int elementX = this.getAbsoluteX();
        int elementY = this.getAbsoluteY();
        int elementWidth = this.getAbsoluteWidth();
        int elementHeight = this.getAbsoluteHeight();

        // Determine which contextual bar to show
        ContextualInfo contextualInfo = getContextualInfoState();

        // Render the appropriate bar
        switch (contextualInfo) {
            case EXPERIENCE:
                renderExperienceBar(graphics, elementX, elementY, elementWidth, elementHeight);
                break;
            case JUMPABLE_VEHICLE:
                renderJumpBar(graphics, elementX, elementY, elementWidth, elementHeight);
                break;
            case LOCATOR:
                renderLocatorBar(graphics, elementX, elementY, elementWidth, elementHeight);
                break;
            default:
                // Render nothing for EMPTY state
                break;
        }
    }

    /**
     * Determines which contextual bar should be displayed based on game state.
     * This mimics the logic from Minecraft 1.21.6's Gui class.
     */
    private ContextualInfo getContextualInfoState() {
        class_746 player = this.minecraft.field_1724;
        if (player == null) return ContextualInfo.EMPTY;

        // In editor mode, show locator bar if always show is enabled, otherwise show experience bar
        if (isEditor()) {
            return alwaysShowLocatorBar ? ContextualInfo.LOCATOR : ContextualInfo.EXPERIENCE;
        }
        
        // If always show locator bar is enabled, return locator (takes priority)
        if (alwaysShowLocatorBar) {
            return ContextualInfo.LOCATOR;
        }
        
        // If always show experience bar is enabled, return experience
        if (alwaysShowExperienceBar) {
            return ContextualInfo.EXPERIENCE;
        }

        boolean hasWaypoints = player.field_3944.method_70936().method_70952();
        boolean hasJumpableVehicle = player.method_45773() != null;
        boolean hasExperience = this.minecraft.field_1761.method_2913();

        if (hasWaypoints) {
            if (hasJumpableVehicle && willPrioritizeJumpInfo()) {
                return ContextualInfo.JUMPABLE_VEHICLE;
            } else {
                return hasExperience && willPrioritizeExperienceInfo() ? ContextualInfo.EXPERIENCE : ContextualInfo.LOCATOR;
            }
        } else if (hasJumpableVehicle) {
            return ContextualInfo.JUMPABLE_VEHICLE;
        } else {
            return hasExperience ? ContextualInfo.EXPERIENCE : ContextualInfo.EMPTY;
        }
    }

    /**
     * Checks if experience info should be prioritized (recently gained experience).
     */
    private boolean willPrioritizeExperienceInfo() {
        class_746 player = this.minecraft.field_1724;
        return player != null && player.field_59964 + EXPERIENCE_BAR_DISPLAY_TICKS > player.field_6012;
    }

    /**
     * Checks if jump info should be prioritized (actively jumping or on cooldown).
     */
    private boolean willPrioritizeJumpInfo() {
        class_746 player = this.minecraft.field_1724;
        if (player == null) return false;
        
        if (player.method_3151() > 0.0F) {
            return true;
        }
        
        class_1316 jumpableVehicle = player.method_45773();
        return jumpableVehicle != null && jumpableVehicle.method_45327() > 0;
    }

    /**
     * Renders the experience bar.
     */
    private void renderExperienceBar(class_332 graphics, int x, int y, int width, int height) {
        var player = this.minecraft.field_1724;
        if (player == null) return;

        // Calculate the color with opacity
        int color = class_9848.method_61324(Math.round(this.opacity * 255f), 255, 255, 255);

        // Only draw the bar if the player requires XP for the next level, or if always show is enabled.
        int xpNeeded = player.method_7349();
        if ((xpNeeded > 0) || alwaysShowExperienceBar || isEditor()) {
            // Calculate the width (in pixels) of the filled portion.
            int filledBarWidth = (int) (player.field_7510 * 183.0f);
            if (isEditor()) filledBarWidth = BAR_WIDTH / 2;

            // Draw the empty (background) experience bar.
            graphics.method_52707(
                class_10799.field_56883,
                EXPERIENCE_BAR_BACKGROUND_SPRITE, 
                x, 
                y, 
                width, 
                height,
                color
            );

            // Draw the filled part of the bar if any XP has been gained.
            if (filledBarWidth > 0) {
                graphics.method_44379(x, y, x + filledBarWidth, y + height);
                graphics.method_52707(
                    class_10799.field_56883,
                    EXPERIENCE_BAR_PROGRESS_SPRITE,
                    x,
                    y,
                    width,
                    height,
                    color
                );
                graphics.method_44380();
            }
        }

        // Render the experience level number if the level is greater than zero, or if always show is enabled.
        if ((player.field_7520 > 0) || alwaysShowExperienceBar || isEditor()) {
            int level = player.field_7520;
            if (isEditor()) level = 42;
            
            class_2561 levelText = class_2561.method_43469("gui.experience.level", level);

            // Center the level text horizontally within the element.
            int textWidth = this.getFont().method_27525(levelText);
            int textX = x + (width - textWidth) / 2;
            int textY = y - 6;

            // Draw shadow around the text for better readability.
            graphics.method_51439(this.getFont(), levelText, textX + 1, textY, SpiffyRenderUtils.colorWithAlpha(0, this.opacity), false);
            graphics.method_51439(this.getFont(), levelText, textX - 1, textY, SpiffyRenderUtils.colorWithAlpha(0, this.opacity), false);
            graphics.method_51439(this.getFont(), levelText, textX, textY + 1, SpiffyRenderUtils.colorWithAlpha(0, this.opacity), false);
            graphics.method_51439(this.getFont(), levelText, textX, textY - 1, SpiffyRenderUtils.colorWithAlpha(0, this.opacity), false);
            // Draw the main level number in yellow (color code 8453920).
            graphics.method_51439(this.getFont(), levelText, textX, textY, SpiffyRenderUtils.colorWithAlpha(8453920, this.opacity), false);
        }
    }

    /**
     * Renders the jump bar for rideable entities.
     */
    private void renderJumpBar(class_332 graphics, int x, int y, int width, int height) {
        class_746 player = this.minecraft.field_1724;
        class_1316 jumpableVehicle = player != null ? player.method_45773() : null;
        if (jumpableVehicle == null && !isEditor()) return;

        int color = class_9848.method_61324(Math.round(this.opacity * 255f), 255, 255, 255);

        // Draw the background
        graphics.method_52707(
            class_10799.field_56883,
            JUMP_BAR_BACKGROUND_SPRITE,
            x,
            y,
            width,
            height,
            color
        );

        if (isEditor()) {
            // Show half-filled jump bar in editor
            int filledWidth = width / 2;
            graphics.method_44379(x, y, x + filledWidth, y + height);
            graphics.method_52707(
                class_10799.field_56883,
                JUMP_BAR_PROGRESS_SPRITE,
                x,
                y,
                width,
                height,
                color
            );
            graphics.method_44380();
        } else if (jumpableVehicle.method_45327() > 0) {
            // Show cooldown overlay
            graphics.method_52707(
                class_10799.field_56883,
                JUMP_BAR_COOLDOWN_SPRITE,
                x,
                y,
                width,
                height,
                color
            );
        } else {
            // Show jump progress
            int filledWidth = (int) (player.method_3151() * 183.0f);
            if (filledWidth > 0) {
                graphics.method_44379(x, y, x + filledWidth, y + height);
                graphics.method_52707(
                    class_10799.field_56883,
                    JUMP_BAR_PROGRESS_SPRITE,
                    x,
                    y,
                    width,
                    height,
                    color
                );
                graphics.method_44380();
            }
        }
    }

    /**
     * Renders the locator bar for waypoints.
     */
    private void renderLocatorBar(class_332 graphics, int x, int y, int width, int height) {
        int color = class_9848.method_61324(Math.round(this.opacity * 255f), 255, 255, 255);

        // Draw the background
        graphics.method_52707(
            class_10799.field_56883,
            LOCATOR_BAR_BACKGROUND,
            x,
            y,
            width,
            height,
            color
        );

        if (isEditor()) {
            // Show some example waypoints in editor mode
            int centerX = x + width / 2;
            int dotY = y - 2;
            
            // Draw 3 example waypoint dots
            for (int i = -1; i <= 1; i++) {
                int dotX = centerX + (i * 30) - DOT_SIZE / 2;
                int dotColor = class_9848.method_61322(color, class_9848.method_61324(255, 100 + i * 50, 150, 200));
                graphics.method_25294(dotX, dotY, dotX + DOT_SIZE, dotY + DOT_SIZE, dotColor);
            }
        } else {
            // Render actual waypoints
            class_1937 level = this.minecraft.method_1560().method_73183();
            this.minecraft.field_1724.field_3944.method_70936().method_70954(this.minecraft.method_1560(), (waypoint) -> {
                if (!isPlayerWaypoint(waypoint)) {
                    double yawAngle = waypoint.method_71153(level, this.minecraft.field_1773.method_19418(), entity -> this.minecraft.field_1773.method_19418().method_55437());
                    if (yawAngle > -VISIBLE_DEGREE_RANGE - 1 && yawAngle <= VISIBLE_DEGREE_RANGE) {
                        renderWaypoint(graphics, waypoint, x, y, width, yawAngle, level);
                    }
                }
            });
        }
    }

    /**
     * Checks if a waypoint belongs to the player.
     */
    private boolean isPlayerWaypoint(class_11200 waypoint) {
        return waypoint.method_70763().left()
            .map(uuid -> uuid.equals(this.minecraft.method_1560().method_5667()))
            .orElse(false);
    }

    /**
     * Renders a single waypoint on the locator bar.
     */
    private void renderWaypoint(class_332 graphics, class_11200 waypoint, int barX, int barY, int barWidth, double yawAngle, class_1937 level) {
        int centerX = barX + barWidth / 2;
        class_11208.class_11209 icon = waypoint.method_70772();
        class_11326 style = this.minecraft.method_71267().method_71326(icon.field_60166);
        float distance = class_3532.method_15355((float) waypoint.method_70773(this.minecraft.method_1560()));
        class_2960 sprite = style.method_71321(distance);
        
        // Calculate waypoint color
        int waypointColor = icon.field_59791.orElseGet(() -> 
            waypoint.method_70763().map(
                left -> class_9848.method_70658(class_9848.method_61330(255, left.hashCode()), 0.9F),
                right -> class_9848.method_70658(class_9848.method_61330(255, right.hashCode()), 0.9F)
            )
        );
        
        // Apply element opacity
        waypointColor = class_9848.method_61322(waypointColor, class_9848.method_61324(Math.round(this.opacity * 255f), 255, 255, 255));
        
        // Calculate position on bar
        int xOffset = (int) (yawAngle * 173.0 / 2.0 / VISIBLE_DEGREE_RANGE);
        int dotX = centerX + xOffset - DOT_SIZE / 2;
        int dotY = barY - 2;
        
        // Draw waypoint dot
        graphics.method_52707(
            class_10799.field_56883,
            sprite,
            dotX,
            dotY,
            DOT_SIZE,
            DOT_SIZE,
            waypointColor
        );
        
        // Draw directional arrow if needed
        class_11200.class_11379 pitchDirection = waypoint.method_71154(level, this.minecraft.field_1773, entity -> this.minecraft.field_1773.method_19418().method_55437());
        if (pitchDirection != class_11200.class_11379.field_60423) {
            int arrowY;
            class_2960 arrowSprite;
            
            if (pitchDirection == class_11200.class_11379.field_60425) {
                arrowY = barY + 6;
                arrowSprite = LOCATOR_BAR_ARROW_DOWN;
            } else {
                arrowY = barY - 6;
                arrowSprite = LOCATOR_BAR_ARROW_UP;
            }
            
            graphics.method_52707(
                class_10799.field_56883,
                arrowSprite,
                dotX + 1,
                arrowY,
                ARROW_WIDTH,
                ARROW_HEIGHT,
                waypointColor
            );
        }
    }

    /**
     * Retrieves the current Minecraft font for rendering text.
     *
     * @return The font used by Minecraft.
     */
    private class_327 getFont() {
        return class_310.method_1551().field_1772;
    }

    @Override
    public int getAbsoluteWidth() {
        return BAR_WIDTH;
    }

    @Override
    public int getAbsoluteHeight() {
        return BAR_HEIGHT;
    }
}
