package de.keksuccino.spiffyhud.customization.elements.vanillalike.hotbar;

import de.keksuccino.fancymenu.customization.element.AbstractElement;
import de.keksuccino.fancymenu.customization.element.ElementBuilder;
import net.minecraft.class_10799;
import net.minecraft.class_1306;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_2960;
import net.minecraft.class_310;
import net.minecraft.class_327;
import net.minecraft.class_332;
import net.minecraft.class_3532;
import net.minecraft.class_9848;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class VanillaLikeHotbarElement extends AbstractElement {

    // Sprite resources for the hotbar in 1.21.5
    private static final class_2960 HOTBAR_SPRITE = class_2960.method_60656("hud/hotbar");
    private static final class_2960 HOTBAR_SELECTION_SPRITE = class_2960.method_60656("hud/hotbar_selection");
    private static final class_2960 HOTBAR_OFFHAND_LEFT_SPRITE = class_2960.method_60656("hud/hotbar_offhand_left");
    private static final class_2960 HOTBAR_OFFHAND_RIGHT_SPRITE = class_2960.method_60656("hud/hotbar_offhand_right");

    private static final int BAR_WIDTH = 182;
    private static final int BAR_HEIGHT = 22;

    public VanillaLikeHotbarElement(@NotNull ElementBuilder<?, ?> builder) {
        super(builder);
    }

    /**
     * Renders the hotbar element.
     * The hotbar background, selection highlight, offhand icons, and slots are all drawn relative to
     * the element's absolute position and size.
     */
    @Override
    public void method_25394(@NotNull class_332 graphics, int mouseX, int mouseY, float partialTick) {
        // Get the absolute position and size of this element.
        int elementX = this.getAbsoluteX();
        int elementY = this.getAbsoluteY();
        int elementWidth = this.getAbsoluteWidth();   // Expected width: 182
        int elementHeight = this.getAbsoluteHeight(); // Expected height: 22

        // Only proceed if the player is available.
        if (class_310.method_1551().field_1724 == null) return;

        // Render the hotbar within the element's bounds.
        this.renderHotbar(graphics, partialTick, elementX, elementY, elementWidth, elementHeight);
    }

    /**
     * Renders the hotbar background, selection highlight, offhand icons, and item slots.
     *
     * @param graphics       The graphics context for rendering.
     * @param partialTick    The partial tick time for animations.
     * @param elementX       The X position of the element.
     * @param elementY       The Y position of the element.
     * @param elementWidth   The width of the element.
     * @param elementHeight  The height of the element.
     */
    private void renderHotbar(class_332 graphics, float partialTick, int elementX, int elementY, int elementWidth, int elementHeight) {
        // Get the current player.
        class_1657 player = this.getCameraPlayer();
        if (player == null) {
            return;
        }

        // Calculate color with opacity
        int color = class_9848.method_61324(Math.round(this.opacity * 255f), 255, 255, 255);

        // Get the player's offhand item and determine the opposite of the main arm.
        class_1799 offhandItem = player.method_6079();
        class_1306 oppositeMainArm = player.method_6068().method_5928();

        // Draw the hotbar background using the element's bounds.
        graphics.method_52707(class_10799.field_56883, HOTBAR_SPRITE, elementX, elementY, elementWidth, elementHeight, color);

        // Draw the selection highlight around the currently selected hotbar slot.
        int selectedSlot = player.method_31548().method_67532();
        // The highlight is drawn with a 1-pixel offset relative to the hotbar background.
        graphics.method_52707(class_10799.field_56883, HOTBAR_SELECTION_SPRITE, elementX - 1 + selectedSlot * 20, elementY - 1, 24, 23, color);

        // Render offhand icons if applicable.
        if (!isEditor()) {
            if (!offhandItem.method_7960()) {
                if (oppositeMainArm == class_1306.field_6182) {
                    // Render offhand icon on the left side of the hotbar.
                    graphics.method_52707(class_10799.field_56883, HOTBAR_OFFHAND_LEFT_SPRITE, elementX - 29, elementY - 1, 29, 24, color);
                } else {
                    // Render offhand icon on the right side of the hotbar.
                    graphics.method_52707(class_10799.field_56883, HOTBAR_OFFHAND_RIGHT_SPRITE, elementX + elementWidth, elementY - 1, 29, 24, color);
                }
            }
        } else {
            // In editor mode, display both offhand icons for demonstration purposes.
            graphics.method_52707(class_10799.field_56883, HOTBAR_OFFHAND_LEFT_SPRITE, elementX - 29, elementY - 1, 29, 24, color);
            graphics.method_52707(class_10799.field_56883, HOTBAR_OFFHAND_RIGHT_SPRITE, elementX + elementWidth, elementY - 1, 29, 24, color);
        }

        // Calculate starting positions for rendering the 9 hotbar slots.
        int slotStartX = elementX + 3; // 3 pixels padding from the left edge.
        int slotY = elementY + 3;      // 3 pixels padding from the top edge.
        int renderSeed = 1;

        // Render each of the 9 main hotbar slots.
        for (int slotIndex = 0; slotIndex < 9; slotIndex++) {
            int slotX = slotStartX + slotIndex * 20; // Each slot is spaced 20 pixels apart.
            renderSlot(graphics, slotX, slotY, partialTick, player, player.method_31548().method_5438(slotIndex), renderSeed++, color);
        }

        // Render the offhand slot if there is an offhand item.
        if (!offhandItem.method_7960()) {
            if (oppositeMainArm == class_1306.field_6182) {
                renderSlot(graphics, elementX - 26, slotY, partialTick, player, offhandItem, renderSeed++, color);
            } else {
                renderSlot(graphics, elementX + elementWidth + 10, slotY, partialTick, player, offhandItem, renderSeed++, color);
            }
        }
    }

    /**
     * Renders a single item slot, including any pop animation and decorations.
     *
     * @param graphics      The graphics context for rendering.
     * @param slotX         The X position of the slot.
     * @param slotY         The Y position of the slot.
     * @param partialTick   The partial tick time for animations.
     * @param player        The current player.
     * @param stack         The item stack to render.
     * @param renderSeed    A seed value for randomized rendering effects.
     * @param color         The color to apply to the rendering.
     */
    private void renderSlot(class_332 graphics, int slotX, int slotY, float partialTick, class_1657 player, class_1799 stack, int renderSeed, int color) {
        // Do not render if the item stack is empty.
        if (stack.method_7960()) {
            return;
        }

        // Determine if the item has a pop animation.
        float popTimeRemaining = (float) stack.method_7965() - partialTick;
        if (popTimeRemaining > 0.0f) {
            // Calculate scaling factor based on the pop animation.
            float scaleFactor = 1.0f + popTimeRemaining / 5.0f;
            graphics.method_51448().pushMatrix();
            // Translate to the center of the slot.
            graphics.method_51448().translate(slotX + 8, slotY + 12);
            // Apply scaling transformation for the pop effect.
            graphics.method_51448().scale(1.0f / scaleFactor, (scaleFactor + 1.0f) / 2.0f);
            // Translate back to the original position.
            graphics.method_51448().translate(-(slotX + 8), -(slotY + 12));
        }

        // Render the item within the slot.
        graphics.method_51423(player, stack, slotX, slotY, renderSeed);

        // If a pop animation was applied, revert the transformation.
        if (popTimeRemaining > 0.0f) {
            graphics.method_51448().popMatrix();
        }

        // Render additional item decorations such as count overlays and cooldown shading.
        renderSlotDecorations(graphics, player, stack, slotX, slotY);

    }

    /**
     * Renders stack count, durability bar, and cooldown overlay for a single slot.
     */
    private void renderSlotDecorations(class_332 graphics, @Nullable class_1657 player, class_1799 stack, int slotX, int slotY) {

        graphics.method_51448().pushMatrix();

        class_310 minecraft = class_310.method_1551();
        class_327 font = minecraft.field_1772;

        if (stack.method_7947() != 1) {
            String countText = String.valueOf(stack.method_7947());
            graphics.method_51433(font, countText, slotX + 19 - 2 - font.method_1727(countText), slotY + 6 + 3, 0xFFFFFF, true);
        }

        if (stack.method_31578()) {
            int barWidth = stack.method_31579();
            int barColor = stack.method_31580();
            int barX = slotX + 2;
            int barY = slotY + 13;
            graphics.method_48196(class_10799.field_56879, barX, barY, barX + 13, barY + 2, 0xFF000000);
            graphics.method_48196(class_10799.field_56879, barX, barY, barX + barWidth, barY + 1, class_9848.method_61334(barColor));
        }

        if (player != null) {
            float cooldownProgress = player.method_7357()
                    .method_7905(stack, minecraft.method_61966().method_60637(true));
            if (cooldownProgress > 0.0F) {
                int overlayTop = slotY + class_3532.method_15375(16.0F * (1.0F - cooldownProgress));
                int overlayBottom = overlayTop + class_3532.method_15386(16.0F * cooldownProgress);
                graphics.method_48196(class_10799.field_56879, slotX, overlayTop, slotX + 16, overlayBottom, Integer.MAX_VALUE);
            }
        }

        graphics.method_51448().popMatrix();
    }

    /**
     * Retrieves the current camera player.
     *
     * @return The current player if available, otherwise null.
     */
    @Nullable
    private class_1657 getCameraPlayer() {
        return (class_310.method_1551().method_1560() instanceof class_1657 p) ? p : null;
    }

    @Override
    public int getAbsoluteWidth() {
        return BAR_WIDTH;
    }

    @Override
    public int getAbsoluteHeight() {
        return BAR_HEIGHT;
    }
}
