package de.keksuccino.spiffyhud.util.level;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import net.minecraft.class_11362;
import net.minecraft.class_1297;
import net.minecraft.class_2203;
import net.minecraft.class_2479;
import net.minecraft.class_2487;
import net.minecraft.class_2495;
import net.minecraft.class_2499;
import net.minecraft.class_2501;
import net.minecraft.class_2514;
import net.minecraft.class_2520;
import net.minecraft.class_8942;
import com.mojang.brigadier.StringReader;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * Utility class for getting NBT data from entities as strings.
 */
public class EntityNbtUtils {

    /**
     * Gets NBT data from an entity as a string using the specified path.
     * For numeric values, returns just the number without type suffix.
     *
     * @param entity The entity to get data from
     * @param path NBT path (like 'Health' or 'Inventory[0].id')
     * @return The data at the specified path as a string, or null if not found
     */
    @Nullable
    public static String getNbtString(@NotNull class_1297 entity, @NotNull String path) {

        try {
            // Save entity data to a compound tag
            class_11362 output = class_11362.method_71459(class_8942.field_60348, entity.method_56673());
            entity.method_5647(output);

            // Parse the NBT path
            class_2203.class_2209 nbtPath = class_2203.method_9360().method_9362(new StringReader(path));

            // Get the data at the path
            List<class_2520> results = nbtPath.method_9366(output.method_71475());

            if (results.isEmpty()) {
                return null;
            }

            class_2520 tag = results.get(0);

            // Special handling for numeric values to remove the type suffix
            if (tag instanceof class_2514 numericTag) {

                String tagString = tag.method_68658().orElse(null);
                Double tagDouble = numericTag.method_68605().orElse(null);
                Long tagLong = numericTag.method_68603().orElse(null);

                if (tagString != null) {
                    // Handle different numeric types to remove suffixes like 'd', 'f', etc.
                    if (tagString.endsWith("d") ||
                            tagString.endsWith("f") ||
                            tagString.endsWith("b") ||
                            tagString.endsWith("s") ||
                            tagString.endsWith("L")) {

                        // For float/double, just use the numeric value without suffix
                        if (tagString.contains(".") && (tagDouble != null)) {
                            return String.valueOf(tagDouble);
                        } else if (tagLong != null) {
                            // For integers, bytes, shorts, longs
                            return String.valueOf(tagLong);
                        }
                    }
                }

            }

            // Default case - return as string
            return tag.method_68658().orElse(null);

        } catch (CommandSyntaxException ignore) {}
        return null;
    }

    /**
     * Gets all possible NBT paths in an entity.
     *
     * @param entity The entity to get paths from
     * @return A list of all NBT paths in the entity
     */
    @NotNull
    public static List<String> getAllNbtPaths(@NotNull class_1297 entity) {
        // Save entity data to a compound tag
        class_11362 output = class_11362.method_71459(class_8942.field_60348, entity.method_56673());
        entity.method_5647(output);

        // Collect all paths
        List<String> paths = new ArrayList<>();
        collectPaths("", output.method_71475(), paths);

        // Sort for better readability
        Collections.sort(paths);
        return paths;
    }

    /**
     * Recursively collects all paths in an NBT tag.
     */
    private static void collectPaths(String prefix, class_2520 tag, List<String> paths) {
        // Add the current path
        if (prefix != null && !prefix.isEmpty()) {
            paths.add(prefix);
        }
        // Recursively collect paths for compound tags
        if (tag instanceof class_2487) {
            class_2487 compound = (class_2487) tag;

            for (String key : compound.method_10541()) {
                String newPrefix = prefix.isEmpty() ? key : prefix + "." + key;
                collectPaths(newPrefix, compound.method_10580(key), paths);
            }
        }
        // Recursively collect paths for list tags
        else if (tag instanceof class_2499) {
            class_2499 list = (class_2499) tag;

            for (int i = 0; i < list.size(); i++) {
                String newPrefix = prefix + "[" + i + "]";
                collectPaths(newPrefix, list.method_10534(i), paths);
            }
        }
        // Handle array tags
        else if (tag instanceof class_2479 || tag instanceof class_2495 || tag instanceof class_2501) {
            int size = 0;
            if (tag instanceof class_2479) {
                size = ((class_2479) tag).method_10521().length;
            } else if (tag instanceof class_2495) {
                size = ((class_2495) tag).method_10588().length;
            } else if (tag instanceof class_2501) {
                size = ((class_2501) tag).method_10615().length;
            }

            for (int i = 0; i < size; i++) {
                paths.add(prefix + "[" + i + "]");
            }
        }
    }

}