package games.enchanted.eg_particle_interactions.common.shapes;

import games.enchanted.eg_particle_interactions.common.util.MathHelpers;
import games.enchanted.eg_particle_interactions.common.util.render.RenderingUtil;
import org.joml.*;

import java.lang.Math;
import net.minecraft.class_4588;

public class QuadFaceShape {
    protected final Vector3dc[] vertices;

    public QuadFaceShape(Vector3dc[] shapeVertices) {
        if(shapeVertices.length % 4 != 0) throw new IllegalArgumentException("VertexShape must be initialised with a multiple of 4 vertices");
        this.vertices = shapeVertices;
    }

    public Vector3dc[] getVertices() {
        return this.vertices;
    }

    /**
     * Adds the geometry for this shape to a {@link class_4588}
     *
     * @param vertexConsumer         the vertex consumer
     * @param uvCoordinates          the uv coordinates for every face, this should have 2 elements defining the uv of the top left and bottom right corners
     * @param pos                    x, y, and z position
     * @param scale                  the scale that the shape will be rendered at
     * @param rotation               the pitch, yaw, and roll that this shape will be rendered at in degrees, rotation is relative to the origin of the shape
     * @param size                   the scale the shape will render at
     * @param lightColour            light colour
     * @param argb                   a colour to tint the shape with, in array of int in argb format
     */
    public void renderShapeWithRotation(class_4588 vertexConsumer, Vector2f[] uvCoordinates, Vector3f pos, Vector3f scale, Vector3f rotation, float size, int lightColour, int[] argb) {
        if(uvCoordinates.length > 2) throw new IllegalArgumentException("VertexShape#renderShape requires exactly 2 elements in uvCoordinates specifying the top left and top right uv coordinates");
        float pitchRad = (float) Math.toRadians(rotation.x);
        float yawRad   = (float) Math.toRadians(rotation.y);
        float rollRad  = (float) Math.toRadians(rotation.z);

        for (int i = 0; i < this.vertices.length; i += 4) {
            Vector3d vertex1 = MathHelpers.rotate3DPoint( new Vector3d(this.vertices[i]    ).mul(scale), pitchRad, yawRad, rollRad ).mul(size).add(pos.x, pos.y, pos.z);
            Vector3d vertex2 = MathHelpers.rotate3DPoint( new Vector3d(this.vertices[i + 1]).mul(scale), pitchRad, yawRad, rollRad ).mul(size).add(pos.x, pos.y, pos.z);
            Vector3d vertex3 = MathHelpers.rotate3DPoint( new Vector3d(this.vertices[i + 2]).mul(scale), pitchRad, yawRad, rollRad ).mul(size).add(pos.x, pos.y, pos.z);
            Vector3d vertex4 = MathHelpers.rotate3DPoint( new Vector3d(this.vertices[i + 3]).mul(scale), pitchRad, yawRad, rollRad ).mul(size).add(pos.x, pos.y, pos.z);

            this.renderVertex(vertexConsumer, vertex1, uvCoordinates[0].x, uvCoordinates[1].y, lightColour, argb);
            this.renderVertex(vertexConsumer, vertex2, uvCoordinates[0].x, uvCoordinates[0].y, lightColour, argb);
            this.renderVertex(vertexConsumer, vertex3, uvCoordinates[1].x, uvCoordinates[0].y, lightColour, argb);
            this.renderVertex(vertexConsumer, vertex4, uvCoordinates[1].x, uvCoordinates[1].y, lightColour, argb);
        }
    }
    /**
     * @see QuadFaceShape#renderShapeWithRotation(class_4588, Vector2f[], Vector3f, Vector3f, Vector3f, float, int, int[])
     *
     */
    public void renderShape(class_4588 vertexConsumer, Vector2f[] uvCoordinates, Vector3f pos, Vector3f scale, float size, int lightColour, int[] argb) {
        this.renderShapeWithRotation(vertexConsumer, uvCoordinates, pos, scale, new Vector3f(0), size, lightColour, argb);
    }

    protected void renderVertex(class_4588 vertexConsumer, Vector3d vertexPos, float u, float v, int lightColor, int[] argb) {
        //? if minecraft: <= 1.21.8 {
        /*RenderingUtil.addVertexToConsumer(vertexConsumer, (float) vertexPos.x, (float) vertexPos.y, (float) vertexPos.z, 0, 0, 1, u, v, lightColor, argb[1] / 255f, argb[2] / 255f, argb[3] / 255f, argb[0] / 255f);
        *///?}
        // TODO: Rendering
    }

    /**
     * Returns an exact copy of a shape
     *
     * @param shapeToCopy the shape to copy
     */
    public static QuadFaceShape copyShape(QuadFaceShape shapeToCopy) {
        Vector3dc[] copiedVerts = new Vector3d[shapeToCopy.getVertices().length];
        for (int i = 0; i < shapeToCopy.getVertices().length; i++) {
            copiedVerts[i] = new Vector3d(shapeToCopy.getVertices()[i]);
        }
        return new QuadFaceShape(copiedVerts);
    }
}
