#version 150

// Light map uniforms - controls lighting
uniform float AmbientLightFactor;
uniform float SkyFactor;
uniform float BlockFactor;
uniform int UseBrightLightmap;
uniform vec3 SkyLightColor;
uniform float NightVisionFactor;
uniform float DarknessScale;
uniform float DarkenWorldFactor;
uniform float BrightnessFactor;

// Input and output
in vec2 texCoord;
out vec4 fragColor;

// Function to get brightness from light level
float getBrightness(float lightLevel) {
    // Make light curve more dramatic
    float shapedLevel = lightLevel / (4.0 - 3.0 * lightLevel);
    
    // Mix with full brightness
    return mix(shapedLevel, 1.0, AmbientLightFactor);
}

// Night vision boost function
void nightVisionBoost(inout vec3 color) {
    // Add brightness for seeing in dark
    color += vec3(0.3, 0.4, 0.3);  // greenish tint
    
    // Keep within bounds
    color = clamp(color, 0.0, 1.0);
}

void main() {
    // Get light levels from texture
    float blockBrightness = getBrightness(texCoord.x) * BlockFactor;
    float skyBrightness = getBrightness(texCoord.y) * SkyFactor;
    
    // Make base color slightly yellowish
    vec3 color = vec3(
        blockBrightness,
        blockBrightness * 0.9 + 0.1,
        blockBrightness * 0.8 + 0.1
    );
    
    // Check if using bright lightmap (End)
    if (UseBrightLightmap != 0) {
        // Make End brighter and more colorful
        color = mix(color, vec3(0.9, 1.0, 0.8), 0.3);
    } else {
        // Add sky lighting
        color += SkyLightColor * skyBrightness * 0.5;
        
        // Darken slightly for drama
        color = mix(color, color * 0.95, 0.1);
        
        // World darkening (boss fights)
        if (DarkenWorldFactor > 0.0) {
            vec3 darkColor = color * vec3(0.8, 0.7, 0.7);
            color = mix(color, darkColor, DarkenWorldFactor);
        }
    }
    
    // Night vision effect
    if (NightVisionFactor > 0.0) {
        nightVisionBoost(color);
    }
    
    // Darkness effect
    if (UseBrightLightmap == 0) {
        color = clamp(color - vec3(DarknessScale * 0.5), 0.0, 1.0);
    }
    
    // Brightness boost - makes slider more effective
    float brightnessMultiplier = 1.0 + (BrightnessFactor * 8.0);
    color *= brightnessMultiplier;
    
    // Keep colors valid
    color = clamp(color, 0.0, 1.0);
    
    fragColor = vec4(color, 1.0);
}