/*
 * Decompiled with CFR 0.152.
 */
package celia.adwadg.ncm.util;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

public class AudioCacheManager {
    private static final String CACHE_DIR = "ncm_cache";
    private static final String META_FILE = "cache_meta.json";
    private static final Gson GSON = new Gson();
    private final Path cacheDirectory;
    private final Path metaFilePath;
    private final Map<String, CacheMetaInfo> cacheMeta;
    private final long maxCacheSize;
    private final long cacheExpireTime;

    public AudioCacheManager(long maxCacheSizeMB, long cacheExpireHours) {
        this.maxCacheSize = maxCacheSizeMB * 1024L * 1024L;
        this.cacheExpireTime = TimeUnit.HOURS.toMillis(cacheExpireHours);
        this.cacheDirectory = Paths.get(".", CACHE_DIR);
        try {
            Files.createDirectories(this.cacheDirectory, new FileAttribute[0]);
        }
        catch (IOException e) {
            System.err.println("\u65e0\u6cd5\u521b\u5efa\u7f13\u5b58\u76ee\u5f55: " + e.getMessage());
        }
        this.metaFilePath = this.cacheDirectory.resolve(META_FILE);
        this.cacheMeta = new ConcurrentHashMap<String, CacheMetaInfo>();
        this.loadCacheMeta();
        this.cleanupExpiredCache();
    }

    private void loadCacheMeta() {
        if (Files.exists(this.metaFilePath, new LinkOption[0])) {
            try (BufferedReader reader = Files.newBufferedReader(this.metaFilePath);){
                JsonObject metaJson = (JsonObject)GSON.fromJson((Reader)reader, JsonObject.class);
                for (String key : metaJson.keySet()) {
                    JsonObject metaInfo = metaJson.getAsJsonObject(key);
                    this.cacheMeta.put(key, new CacheMetaInfo(metaInfo.get("size").getAsLong(), metaInfo.get("lastAccess").getAsLong(), metaInfo.get("createTime").getAsLong()));
                }
            }
            catch (IOException e) {
                System.err.println("\u52a0\u8f7d\u7f13\u5b58\u5143\u6570\u636e\u5931\u8d25: " + e.getMessage());
            }
        }
    }

    private void saveCacheMeta() {
        try (BufferedWriter writer = Files.newBufferedWriter(this.metaFilePath, new OpenOption[0]);){
            JsonObject metaJson = new JsonObject();
            for (Map.Entry<String, CacheMetaInfo> entry : this.cacheMeta.entrySet()) {
                JsonObject metaInfo = new JsonObject();
                metaInfo.addProperty("size", (Number)entry.getValue().size);
                metaInfo.addProperty("lastAccess", (Number)entry.getValue().lastAccess);
                metaInfo.addProperty("createTime", (Number)entry.getValue().createTime);
                metaJson.add(entry.getKey(), (JsonElement)metaInfo);
            }
            GSON.toJson((JsonElement)metaJson, (Appendable)writer);
        }
        catch (IOException e) {
            System.err.println("\u4fdd\u5b58\u7f13\u5b58\u5143\u6570\u636e\u5931\u8d25: " + e.getMessage());
        }
    }

    public boolean isCached(String cacheKey) {
        Path cacheFile = this.getCacheFilePath(cacheKey);
        return Files.exists(cacheFile, new LinkOption[0]) && !this.isExpired(cacheKey);
    }

    public InputStream getCachedAudio(String cacheKey) throws IOException {
        if (!this.isCached(cacheKey)) {
            return null;
        }
        CacheMetaInfo metaInfo = this.cacheMeta.get(cacheKey);
        if (metaInfo != null) {
            metaInfo.lastAccess = System.currentTimeMillis();
            this.saveCacheMeta();
        }
        return Files.newInputStream(this.getCacheFilePath(cacheKey), new OpenOption[0]);
    }

    public void cacheAudio(String cacheKey, InputStream audioStream, long contentLength) throws IOException {
        Path cacheFile = this.getCacheFilePath(cacheKey);
        this.ensureCacheSpace(contentLength);
        try (OutputStream out = Files.newOutputStream(cacheFile, new OpenOption[0]);){
            int bytesRead;
            byte[] buffer = new byte[8192];
            long totalBytes = 0L;
            while ((bytesRead = audioStream.read(buffer)) != -1) {
                out.write(buffer, 0, bytesRead);
                totalBytes += (long)bytesRead;
            }
            long currentTime = System.currentTimeMillis();
            this.cacheMeta.put(cacheKey, new CacheMetaInfo(totalBytes, currentTime, currentTime));
            this.saveCacheMeta();
        }
    }

    private void ensureCacheSpace(long requiredSpace) throws IOException {
        long[] currentSize = new long[]{this.getCurrentCacheSize()};
        if (currentSize[0] + requiredSpace > this.maxCacheSize) {
            this.cacheMeta.entrySet().stream().sorted((e1, e2) -> Long.compare(((CacheMetaInfo)e1.getValue()).lastAccess, ((CacheMetaInfo)e2.getValue()).lastAccess)).forEachOrdered(entry -> {
                if (currentSize[0] + requiredSpace > this.maxCacheSize) {
                    try {
                        Files.deleteIfExists(this.getCacheFilePath((String)entry.getKey()));
                        this.cacheMeta.remove(entry.getKey());
                        currentSize[0] = currentSize[0] - ((CacheMetaInfo)entry.getValue()).size;
                    }
                    catch (IOException e) {
                        System.err.println("\u5220\u9664\u7f13\u5b58\u6587\u4ef6\u5931\u8d25: " + e.getMessage());
                    }
                }
            });
            this.saveCacheMeta();
        }
    }

    private long getCurrentCacheSize() {
        return this.cacheMeta.values().stream().mapToLong(meta -> meta.size).sum();
    }

    private boolean isExpired(String cacheKey) {
        CacheMetaInfo metaInfo = this.cacheMeta.get(cacheKey);
        if (metaInfo == null) {
            return true;
        }
        return System.currentTimeMillis() - metaInfo.createTime > this.cacheExpireTime;
    }

    public void cleanupExpiredCache() {
        this.cacheMeta.entrySet().removeIf(entry -> {
            if (this.isExpired((String)entry.getKey())) {
                try {
                    Files.deleteIfExists(this.getCacheFilePath((String)entry.getKey()));
                    return true;
                }
                catch (IOException e) {
                    System.err.println("\u5220\u9664\u8fc7\u671f\u7f13\u5b58\u6587\u4ef6\u5931\u8d25: " + e.getMessage());
                    return false;
                }
            }
            return false;
        });
        this.saveCacheMeta();
    }

    public void clearCache() {
        try {
            Files.walk(this.cacheDirectory, new FileVisitOption[0]).filter(path -> !path.equals(this.cacheDirectory) && !path.equals(this.metaFilePath)).forEach(path -> {
                try {
                    Files.deleteIfExists(path);
                }
                catch (IOException e) {
                    System.err.println("\u5220\u9664\u7f13\u5b58\u6587\u4ef6\u5931\u8d25: " + e.getMessage());
                }
            });
            this.cacheMeta.clear();
            this.saveCacheMeta();
        }
        catch (IOException e) {
            System.err.println("\u6e05\u7406\u7f13\u5b58\u5931\u8d25: " + e.getMessage());
        }
    }

    public JsonObject getCacheStatus() {
        JsonObject status = new JsonObject();
        status.addProperty("enabled", Boolean.valueOf(true));
        status.addProperty("totalFiles", (Number)this.cacheMeta.size());
        status.addProperty("totalSizeMB", (Number)(this.getCurrentCacheSize() / 0x100000L));
        status.addProperty("maxSizeMB", (Number)(this.maxCacheSize / 0x100000L));
        status.addProperty("expireHours", (Number)TimeUnit.MILLISECONDS.toHours(this.cacheExpireTime));
        Runtime runtime = Runtime.getRuntime();
        long usedMemory = runtime.totalMemory() - runtime.freeMemory();
        long maxMemory = runtime.maxMemory();
        status.addProperty("usedMemoryMB", (Number)(usedMemory / 0x100000L));
        status.addProperty("maxMemoryMB", (Number)(maxMemory / 0x100000L));
        status.addProperty("memoryUsagePercent", (Number)((int)(usedMemory * 100L / maxMemory)));
        return status;
    }

    private Path getCacheFilePath(String cacheKey) {
        return this.cacheDirectory.resolve(cacheKey + ".mp3");
    }

    public static String generateCacheKey(long songId, int bitrate) {
        return "song_" + songId + "_" + bitrate;
    }

    private static class CacheMetaInfo {
        long size;
        long lastAccess;
        long createTime;

        CacheMetaInfo(long size, long lastAccess, long createTime) {
            this.size = size;
            this.lastAccess = lastAccess;
            this.createTime = createTime;
        }
    }
}

