/*
 * Decompiled with CFR 0.152.
 */
package com.serveroptimiser.optimization;

import com.serveroptimiser.ServerOptimiserPlugin;
import com.serveroptimiser.config.ConfigManager;
import com.serveroptimiser.utils.PerformanceMonitor;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import org.bukkit.Chunk;
import org.bukkit.World;
import org.bukkit.entity.Entity;
import org.bukkit.entity.Player;
import org.bukkit.permissions.ServerOperator;
import org.bukkit.plugin.Plugin;
import org.bukkit.scheduler.BukkitRunnable;

public class LagDetector {
    private final ServerOptimiserPlugin plugin;
    private final ConfigManager config;
    private final PerformanceMonitor performanceMonitor;
    private final Map<Chunk, Integer> laggyChunks;
    private final Set<Entity> laggyEntities;
    private final AtomicInteger lagDetections;
    private boolean isMonitoring;
    private long lastLagAlert;

    public LagDetector(ServerOptimiserPlugin plugin, ConfigManager config, PerformanceMonitor performanceMonitor) {
        this.plugin = plugin;
        this.config = config;
        this.performanceMonitor = performanceMonitor;
        this.laggyChunks = new ConcurrentHashMap<Chunk, Integer>();
        this.laggyEntities = ConcurrentHashMap.newKeySet();
        this.lagDetections = new AtomicInteger(0);
        this.isMonitoring = false;
        this.lastLagAlert = 0L;
    }

    public void startMonitoring() {
        if (this.isMonitoring) {
            return;
        }
        this.isMonitoring = true;
        new BukkitRunnable(){

            public void run() {
                if (!LagDetector.this.isMonitoring) {
                    this.cancel();
                    return;
                }
                LagDetector.this.detectLagSources();
            }
        }.runTaskTimer((Plugin)this.plugin, 0L, (long)this.config.getLagCheckInterval() * 20L);
    }

    public void stopMonitoring() {
        this.isMonitoring = false;
    }

    private void detectLagSources() {
        final double currentTPS = this.performanceMonitor.getCurrentTPS();
        if (currentTPS >= this.config.getLagThreshold()) {
            return;
        }
        this.lagDetections.incrementAndGet();
        new BukkitRunnable(){

            public void run() {
                LagDetector.this.detectLaggyChunks();
                LagDetector.this.detectLaggyEntities();
                if (LagDetector.this.config.isAlertAdmins()) {
                    LagDetector.this.alertAdmins(currentTPS);
                }
                if (LagDetector.this.config.isAutoKillLagSources()) {
                    LagDetector.this.handleLagSources();
                }
            }
        }.runTask((Plugin)this.plugin);
    }

    private void detectLaggyChunks() {
        for (World world : this.plugin.getServer().getWorlds()) {
            for (Chunk chunk : world.getLoadedChunks()) {
                try {
                    int entityCount = chunk.getEntities().length;
                    int tileEntityCount = chunk.getTileEntities().length;
                    int lagScore = entityCount + tileEntityCount * 2;
                    if (lagScore <= 100) continue;
                    this.laggyChunks.put(chunk, lagScore);
                }
                catch (IllegalStateException e) {
                    // empty catch block
                }
            }
        }
    }

    private void detectLaggyEntities() {
        for (World world : this.plugin.getServer().getWorlds()) {
            for (Entity entity : world.getEntities()) {
                if (!this.isEntityCausingLag(entity)) continue;
                this.laggyEntities.add(entity);
            }
        }
    }

    private boolean isEntityCausingLag(Entity entity) {
        if (entity instanceof Player) {
            return false;
        }
        List nearbyEntities = entity.getNearbyEntities(5.0, 5.0, 5.0);
        if (nearbyEntities.size() > 50) {
            return true;
        }
        if (entity.getTicksLived() > 72000) {
            return entity.getVelocity().lengthSquared() < 0.01;
        }
        return false;
    }

    private void alertAdmins(double currentTPS) {
        long currentTime = System.currentTimeMillis();
        if (currentTime - this.lastLagAlert < 300000L) {
            return;
        }
        this.lastLagAlert = currentTime;
        String message = String.format("\u00a7c[ServerOptimiser] Lag detected! TPS: %.2f | Laggy chunks: %d | Laggy entities: %d", currentTPS, this.laggyChunks.size(), this.laggyEntities.size());
        this.plugin.getServer().getOnlinePlayers().stream().filter(ServerOperator::isOp).forEach(player -> player.sendMessage(message));
        this.plugin.getLogger().warning(message.replace("\u00a7c", ""));
    }

    private void handleLagSources() {
        new BukkitRunnable(){

            public void run() {
                int entitiesRemoved = 0;
                Iterator<Entity> entityIterator = LagDetector.this.laggyEntities.iterator();
                while (entityIterator.hasNext()) {
                    Entity entity = entityIterator.next();
                    if (!entity.isValid() || entity instanceof Player) continue;
                    entity.remove();
                    ++entitiesRemoved;
                    entityIterator.remove();
                }
                for (Map.Entry<Chunk, Integer> entry : LagDetector.this.laggyChunks.entrySet()) {
                    Chunk chunk = entry.getKey();
                    if (!chunk.isLoaded()) continue;
                    LagDetector.this.optimizeLaggyChunk(chunk);
                }
                if (entitiesRemoved > 0 && LagDetector.this.config.isVerboseOutput()) {
                    LagDetector.this.plugin.getLogger().info("Auto-lag handler removed " + entitiesRemoved + " laggy entities");
                }
                LagDetector.this.laggyChunks.clear();
                LagDetector.this.laggyEntities.clear();
            }
        }.runTask((Plugin)this.plugin);
    }

    private void optimizeLaggyChunk(Chunk chunk) {
        Entity[] entities = chunk.getEntities();
        int removeCount = Math.max(0, entities.length - 50);
        int removed = 0;
        for (Entity entity : entities) {
            if (removed >= removeCount) break;
            if (entity instanceof Player || entity.isPersistent() || entity.getCustomName() != null || entity.getTicksLived() <= 1200) continue;
            entity.remove();
            ++removed;
        }
    }

    public void triggerLagDetection() {
        new BukkitRunnable(){

            public void run() {
                LagDetector.this.detectLagSources();
                LagDetector.this.plugin.getServer().getOnlinePlayers().stream().filter(ServerOperator::isOp).forEach(player -> player.sendMessage("\u00a7a[ServerOptimiser] Manual lag detection completed."));
            }
        }.runTaskAsynchronously((Plugin)this.plugin);
    }

    public String getLagReport() {
        double currentTPS = this.performanceMonitor.getCurrentTPS();
        return String.format("\u00a7eLag Report:\n\u00a77Current TPS: \u00a7f%.2f\n\u00a77Laggy Chunks: \u00a7f%d\n\u00a77Laggy Entities: \u00a7f%d\n\u00a77Total Detections: \u00a7f%d\n\u00a77Monitoring: \u00a7f%s", currentTPS, this.laggyChunks.size(), this.laggyEntities.size(), this.lagDetections.get(), this.isMonitoring ? "ON" : "OFF");
    }

    public Map<Chunk, Integer> getLaggyChunks() {
        return new HashMap<Chunk, Integer>(this.laggyChunks);
    }

    public Set<Entity> getLaggyEntities() {
        return new HashSet<Entity>(this.laggyEntities);
    }

    public int getTotalLagDetections() {
        return this.lagDetections.get();
    }

    public boolean isMonitoring() {
        return this.isMonitoring;
    }

    public void resetStats() {
        this.lagDetections.set(0);
        this.laggyChunks.clear();
        this.laggyEntities.clear();
    }
}

