/*
 * Decompiled with CFR 0.152.
 */
package com.serveroptimiser.optimization;

import com.serveroptimiser.ServerOptimiserPlugin;
import com.serveroptimiser.config.ConfigManager;
import com.serveroptimiser.utils.PerformanceMonitor;
import java.util.concurrent.atomic.AtomicInteger;
import org.bukkit.GameRule;
import org.bukkit.plugin.Plugin;
import org.bukkit.scheduler.BukkitRunnable;
import org.bukkit.scheduler.BukkitTask;

public class TPSManager {
    private final ServerOptimiserPlugin plugin;
    private final ConfigManager config;
    private final PerformanceMonitor performanceMonitor;
    private final AtomicInteger adjustmentsMade;
    private double currentTPSTarget;
    private boolean isThrottling;
    private boolean isManuallyLimited;
    private BukkitTask tpsLimiterTask;
    private long lastTickTime;

    public TPSManager(ServerOptimiserPlugin plugin, ConfigManager config, PerformanceMonitor performanceMonitor) {
        this.plugin = plugin;
        this.config = config;
        this.performanceMonitor = performanceMonitor;
        this.adjustmentsMade = new AtomicInteger(0);
        this.currentTPSTarget = config.getMaxTPS();
        this.isThrottling = false;
        this.isManuallyLimited = false;
        this.lastTickTime = System.currentTimeMillis();
    }

    public void monitorAndAdjustTPS() {
        new BukkitRunnable(){

            public void run() {
                double currentTPS = TPSManager.this.performanceMonitor.getCurrentTPS();
                if (TPSManager.this.config.isAutoTickAdjustment()) {
                    TPSManager.this.adjustTPSBasedOnPerformance(currentTPS);
                }
                if (TPSManager.this.config.isVerboseOutput() && TPSManager.this.adjustmentsMade.get() % 20 == 0) {
                    TPSManager.this.plugin.getLogger().info(String.format("Current TPS: %.2f, Target: %.2f, Throttling: %s", currentTPS, TPSManager.this.currentTPSTarget, TPSManager.this.isThrottling ? "ON" : "OFF"));
                }
            }
        }.runTaskTimer((Plugin)this.plugin, 0L, (long)this.config.getTickCheckInterval() * 20L);
    }

    private void adjustTPSBasedOnPerformance(double currentTPS) {
        if (this.isManuallyLimited) {
            return;
        }
        double minTPS = this.config.getMinTPS();
        double maxTPS = this.config.getMaxTPS();
        if (currentTPS < minTPS && !this.isThrottling) {
            this.startTPSThrottling();
        } else if (currentTPS > this.currentTPSTarget + 1.0 && this.isThrottling) {
            this.reduceTPSThrottling();
        } else if (currentTPS < minTPS - 2.0 && this.isThrottling) {
            this.increaseTPSThrottling();
        } else if (currentTPS >= maxTPS - 1.0 && this.isThrottling) {
            this.stopTPSThrottling();
        }
        this.adjustmentsMade.incrementAndGet();
    }

    private void startTPSThrottling() {
        this.isThrottling = true;
        this.currentTPSTarget = Math.max(this.config.getMinTPS(), 15.0);
        this.scheduleTickReduction();
        if (this.config.isVerboseOutput()) {
            this.plugin.getLogger().info("TPS throttling activated. Target TPS: " + this.currentTPSTarget);
        }
    }

    private void increaseTPSThrottling() {
        if (this.currentTPSTarget > this.config.getMinTPS()) {
            this.currentTPSTarget = Math.max(this.config.getMinTPS(), this.currentTPSTarget - 1.0);
            this.scheduleTickReduction();
            if (this.config.isVerboseOutput()) {
                this.plugin.getLogger().info("Increased TPS throttling. Target TPS: " + this.currentTPSTarget);
            }
        }
    }

    private void reduceTPSThrottling() {
        if (this.currentTPSTarget < this.config.getMaxTPS()) {
            this.currentTPSTarget = Math.min(this.config.getMaxTPS(), this.currentTPSTarget + 0.5);
            if (this.config.isVerboseOutput()) {
                this.plugin.getLogger().info("Reduced TPS throttling. Target TPS: " + this.currentTPSTarget);
            }
        }
    }

    private void stopTPSThrottling() {
        this.isThrottling = false;
        this.currentTPSTarget = this.config.getMaxTPS();
        this.scheduleTickRestoration();
        if (this.config.isVerboseOutput()) {
            this.plugin.getLogger().info("TPS throttling deactivated. Returning to normal operation.");
        }
    }

    private void scheduleTickReduction() {
        new BukkitRunnable(){

            public void run() {
                TPSManager.this.plugin.getServer().getWorlds().forEach(world -> {
                    Integer currentRandomTickSpeed = (Integer)world.getGameRuleValue(GameRule.RANDOM_TICK_SPEED);
                    if (currentRandomTickSpeed != null && currentRandomTickSpeed > 1) {
                        world.setGameRule(GameRule.RANDOM_TICK_SPEED, (Object)Math.max(1, currentRandomTickSpeed - 1));
                    }
                });
                TPSManager.this.plugin.getEntityOptimizer().optimizeEntities();
            }
        }.runTask((Plugin)this.plugin);
    }

    private void scheduleTickRestoration() {
        new BukkitRunnable(){

            public void run() {
                TPSManager.this.plugin.getServer().getWorlds().forEach(world -> world.setGameRule(GameRule.RANDOM_TICK_SPEED, (Object)3));
            }
        }.runTask((Plugin)this.plugin);
    }

    public boolean setTPSTarget(double target) {
        if (target < this.config.getMinTPS() || target > this.config.getMaxTPS()) {
            return false;
        }
        this.currentTPSTarget = target;
        this.stopTPSLimiter();
        if (target < 20.0) {
            this.startTPSLimiter(target);
            this.isManuallyLimited = true;
        } else {
            this.isManuallyLimited = false;
            this.stopTPSThrottling();
        }
        if (this.config.isVerboseOutput() && Math.abs(target - 20.0) > 2.0) {
            this.plugin.getLogger().info(String.format("TPS manually set to %.1f", target));
        }
        return true;
    }

    public void forceOptimization() {
        if (!this.isManuallyLimited) {
            this.startTPSThrottling();
        }
        this.plugin.getEntityOptimizer().optimizeEntities();
        this.plugin.getChunkManager().optimizeChunks();
        this.adjustmentsMade.incrementAndGet();
        if (this.config.isVerboseOutput()) {
            this.plugin.getLogger().info("Forced TPS optimization executed.");
        }
    }

    public double getCurrentTPSTarget() {
        return this.currentTPSTarget;
    }

    public boolean isThrottling() {
        return this.isThrottling;
    }

    public int getTotalAdjustmentsMade() {
        return this.adjustmentsMade.get();
    }

    public void resetStats() {
        this.adjustmentsMade.set(0);
    }

    private void startTPSLimiter(final double targetTPS) {
        if (this.tpsLimiterTask != null && !this.tpsLimiterTask.isCancelled()) {
            this.tpsLimiterTask.cancel();
        }
        final long targetTickTime = (long)(1000.0 / targetTPS);
        this.tpsLimiterTask = new BukkitRunnable(){

            public void run() {
                long currentTime = System.currentTimeMillis();
                long timeSinceLastTick = currentTime - TPSManager.this.lastTickTime;
                if (timeSinceLastTick < targetTickTime) {
                    try {
                        Thread.sleep(targetTickTime - timeSinceLastTick);
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                    }
                }
                TPSManager.this.lastTickTime = System.currentTimeMillis();
                if (targetTPS < 15.0) {
                    TPSManager.this.scheduleTickReduction();
                }
            }
        }.runTaskTimer((Plugin)this.plugin, 0L, 1L);
        if (this.config.isVerboseOutput() && targetTPS < 18.0) {
            this.plugin.getLogger().info(String.format("TPS limiter started - Target: %.1f TPS", targetTPS));
        }
    }

    private void stopTPSLimiter() {
        if (this.tpsLimiterTask != null && !this.tpsLimiterTask.isCancelled()) {
            this.tpsLimiterTask.cancel();
            this.tpsLimiterTask = null;
        }
        this.isManuallyLimited = false;
        if (this.config.isVerboseOutput() && this.isManuallyLimited) {
            this.plugin.getLogger().info("TPS limiter stopped");
        }
    }

    public void resetTPS() {
        this.stopTPSLimiter();
        this.stopTPSThrottling();
        this.currentTPSTarget = 20.0;
        if (this.config.isVerboseOutput() && (this.isManuallyLimited || this.isThrottling)) {
            this.plugin.getLogger().info("TPS reset to normal (20 TPS)");
        }
    }

    public boolean isManuallyLimited() {
        return this.isManuallyLimited;
    }
}

