/*
    @author Hertz
    @version 1.0
*/

let ClientboundSetEntityMotionPacket = Java.loadClass('net.minecraft.network.protocol.game.ClientboundSetEntityMotionPacket');

let $ClipContext = Java.loadClass('net.minecraft.world.level.ClipContext')
let $ProjectileUtil = Java.loadClass('net.minecraft.world.entity.projectile.ProjectileUtil')

/**
 * 
 * @param {Internal.LivingEntity} entity 
 * @param {Internal.ServerLevel} level 
 * @param {number} distance
 * @returns {Object}
 * Returns the block and/or entity that the entity is looking at.
 * Ignores non-solid blocks and spectators.
 */
let advancedRayTrace = (entity, level, distance) => {
    let eyePos = entity.eyePosition;
    let viewVec = entity.getViewVector(1)
    let endPos = eyePos.add(viewVec.x() * distance, viewVec.y() * distance, viewVec.z() * distance)
    let aabb = AABB.of(eyePos.x(), eyePos.y(), eyePos.z(), endPos.x(), endPos.y(), endPos.z())

    let ray = $ProjectileUtil.getEntityHitResult(level, entity, eyePos, endPos, aabb, (e) => {
        return !e.isSpectator()
    }, 0)

    let clip = new $ClipContext(
        entity.getEyePosition(1), 
        entity.getEyePosition(1).add(entity.getLookAngle().scale(distance)), 
        'collider', 'none', 
        entity
    )
    let hit = level.clip(clip)
    if (ray == null) {
        return {
            block: hit.getBlockPos() ? level.getBlock(hit.getBlockPos()) : null,
            entity: null
        }
    }
    return {
        block: level.getBlock(hit.getBlockPos()),
        entity: ray.entity
    }
}

function clamp(value, min, max) {
    return Math.max(min, Math.min(max, value));
}

function doVectorMath(player, entity, mult, grip_distance, vert_offset) {
    let lookAngle = player.getLookAngle().scale(grip_distance)
    let playerLocation = Vec3d(
        player.getX(),
        (player.getY() + 1.0),
        player.getZ()
    )
    let targetLocation = Vec3d(
        (playerLocation.x() + lookAngle.x()),
        ((playerLocation.y() + lookAngle.y()) + vert_offset),
        (playerLocation.z() + lookAngle.z())
    )
    let Target = Vec3d(
        (entity.getX() - targetLocation.x()),
        (entity.getY() - targetLocation.y()),
        (entity.getZ() - targetLocation.z())
    ).scale(-1)
    entity.setNoGravity(true)
    entity.resetFallDistance()
    let mot = Target.scale(Target.length() * mult)
    // entity.getServer().tell(mot)
    mot = Vec3d(
        clamp(mot.x(), -6, 6),
        clamp(mot.y(), -6, 6),
        clamp(mot.z(), -6, 6)
    );
    // entity.getServer().tell(mot)
    entity.setDeltaMovement(mot)
    if (entity.isPlayer()) {
        entity.connection.send(new ClientboundSetEntityMotionPacket(entity));
    }
}

function resolveAllegedBooleanFromObject(thing) {
    if (thing.toString() == 'true') { return true; }
    if (thing.toString() == 'false') { return false; }
    return null
}

var targetSet = new Set();

StartupEvents.registry('palladium:abilities', (event) => {

    event.create('arrzenhanced:telekinesis')
    .icon(palladium.createItemIcon('minecraft:ender_pearl'))
    .addProperty('multiplier', 'float', 0.12, 'Multiplier for pull/push')
    .addProperty('max_distance', 'integer', 8, 'Max distance to select targets at')
    .addProperty('grip_distance', 'integer', 5, 'Distance to hold targets at')
    .addProperty('fail_chance', 'float', 0.5, 'Chance to cancel telekinesis on damage (0.5 == 50%)')
    .addProperty('particles', 'boolean', true, 'Spawn telekinesis particles')
    .addProperty('vert_offset', 'float', 0.0, 'Vertical (Y-val) offset for where to hold the entity')
    .addProperty('collect', 'boolean', false, 'If we should collect entities while telekinesis is active')
    .firstTick((entity, entry, holder, enabled) => {
        if (enabled) {
            var multiplier = entry.getPropertyByName('multiplier');
            var max_distance = entry.getPropertyByName('max_distance');
            var grip_distance = entry.getPropertyByName('grip_distance');
            let target = advancedRayTrace(entity, entity.getLevel(), max_distance).entity
            if (target) {
                targetSet.add(target)
            }
            targetSet.forEach(storedTarget => {
                doVectorMath(entity, storedTarget, multiplier, grip_distance)
            })
        }
    })
    .tick((entity, entry, holder, enabled) => {
        if (enabled) {
            var multiplier = entry.getPropertyByName('multiplier');
            var max_distance = entry.getPropertyByName('max_distance');
            var grip_distance = entry.getPropertyByName('grip_distance');
            var vert_offset = entry.getPropertyByName('vert_offset');
            var collect = resolveAllegedBooleanFromObject(entry.getPropertyByName('collect'))
            var spawnParticles = resolveAllegedBooleanFromObject(entry.getPropertyByName('particles'))
            if (collect) {
                let target = advancedRayTrace(entity, entity.getLevel(), max_distance).entity
                if (target) {
                    targetSet.add(target)
                }
            }
            targetSet.forEach(storedTarget => {
                let player = storedTarget
                let level = player.getLevel()
                if (!storedTarget) {return}
                if (Math.sqrt(entity.distanceToSqr(storedTarget)) > 20) {
                    storedTarget.setNoGravity(false)
                    targetSet.delete(`${storedTarget}`)
                    return
                }
                if (spawnParticles) {
                    let radius = 1.05
                    let particle = Utils.particleOptions('dust 0 0 0 1')
                    let particleCount = 20

                    for (let i = 0; i < particleCount; i++) {
                        let theta = Math.acos(2 * Math.random() - 1); // Polar angle
                        let phi = 2 * JavaMath.PI * Math.random(); // Azimuthal angle
                
                        // Spherical to Cartesian conversion
                        let xOffset = radius * Math.sin(theta) * Math.cos(phi);
                        let yOffset = radius * Math.sin(theta) * Math.sin(phi);
                        let zOffset = radius * Math.cos(theta);
                
                        // Spawn the particle at the calculated offset
                        level.spawnParticles(
                            particle,
                            true,
                            player.x + xOffset,
                            player.y + yOffset + 1, // Adjust for player's height
                            player.z + zOffset,
                            0, 0, 0, // No velocity for static particles
                            1, // Count
                            0 // ?
                        );
                    }
                }
                doVectorMath(entity, storedTarget, multiplier, grip_distance, vert_offset)
            })
        };
    })
    .lastTick((entity, entry, holder, enabled) => {
        targetSet.forEach(storedTarget => {
            storedTarget.setNoGravity(false)
        })
        targetSet.clear()
    })
});