/*
 * Decompiled with CFR 0.152.
 */
package com.quickskin.mod.client.services;

import com.mojang.blaze3d.platform.NativeImage;
import com.quickskin.mod.QuickSkin;
import com.quickskin.mod.common.data.AnimationMetadata;
import com.quickskin.mod.common.data.AssetMetadata;
import com.quickskin.mod.common.data.SkinPreferences;
import com.quickskin.mod.common.data.SkinResolution;
import com.quickskin.mod.common.data.TextureQuality;
import com.quickskin.mod.common.util.HDTextureProcessor;
import com.quickskin.mod.common.util.HashUtil;
import com.quickskin.mod.common.util.SkinModelDetector;
import com.quickskin.mod.common.util.StbGifLoader;
import com.quickskin.mod.config.ClientConfig;
import com.quickskin.mod.platform.PlatformHelper;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Stream;
import javax.imageio.ImageIO;
import net.minecraft.client.Minecraft;
import net.minecraft.client.renderer.texture.AbstractTexture;
import net.minecraft.client.renderer.texture.DynamicTexture;
import net.minecraft.resources.ResourceLocation;
import net.neoforged.api.distmarker.Dist;
import net.neoforged.api.distmarker.OnlyIn;
import org.jetbrains.annotations.Nullable;

@OnlyIn(value=Dist.CLIENT)
public class LocalAssetManager {
    private static LocalAssetManager instance;
    private final Map<String, AssetMetadata> metadataCache = new ConcurrentHashMap<String, AssetMetadata>();
    private final Map<String, Path> hashToSourcePath = new ConcurrentHashMap<String, Path>();
    private final Map<String, Map<TextureQuality, ResourceLocation>> textureRegistry = new ConcurrentHashMap<String, Map<TextureQuality, ResourceLocation>>();
    private Path skinsDirectory;
    private Path capesDirectory;
    private Path cacheDirectory;
    private SkinPreferences skinPreferences;
    private Path preferencesFile;

    private LocalAssetManager() {
    }

    public static LocalAssetManager getInstance() {
        if (instance == null) {
            instance = new LocalAssetManager();
        }
        return instance;
    }

    public void init() {
        QuickSkin.LOGGER.info("Initializing LocalAssetManager...");
        this.skinsDirectory = PlatformHelper.getSkinsDirectory();
        this.capesDirectory = PlatformHelper.getCapesDirectory();
        this.cacheDirectory = PlatformHelper.getCacheDirectory();
        try {
            Files.createDirectories(this.skinsDirectory, new FileAttribute[0]);
            Files.createDirectories(this.capesDirectory, new FileAttribute[0]);
            Files.createDirectories(this.cacheDirectory, new FileAttribute[0]);
        }
        catch (IOException e) {
            QuickSkin.LOGGER.error("Failed to create asset directories", (Throwable)e);
        }
        this.preferencesFile = PlatformHelper.getConfigDirectory().resolve("skin-preferences.json");
        this.skinPreferences = SkinPreferences.load(this.preferencesFile);
        QuickSkin.LOGGER.info("Loaded {} skin preferences", (Object)this.skinPreferences.size());
        this.discoverLocalAssets();
        QuickSkin.LOGGER.info("LocalAssetManager initialized with {} assets", (Object)this.metadataCache.size());
    }

    public void discoverLocalAssets() {
        QuickSkin.LOGGER.info("Discovering local assets...");
        this.metadataCache.clear();
        this.hashToSourcePath.clear();
        int skinsFound = this.scanDirectory(this.skinsDirectory, "skin");
        QuickSkin.LOGGER.info("Found {} skins", (Object)skinsFound);
        int capesFound = this.scanDirectory(this.capesDirectory, "cape");
        QuickSkin.LOGGER.info("Found {} capes", (Object)capesFound);
    }

    private int scanDirectory(Path directory, String type) {
        if (!Files.exists(directory, new LinkOption[0])) {
            return 0;
        }
        int count = 0;
        try (Stream<Path> paths = Files.walk(directory, new FileVisitOption[0]);){
            for (Path path : paths.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).toList()) {
                AssetMetadata metadata;
                String fileName = path.getFileName().toString().toLowerCase();
                if (fileName.endsWith(".png")) {
                    metadata = this.processPngAsset(path, type);
                    if (metadata == null) continue;
                    this.metadataCache.put(metadata.hash(), metadata);
                    this.hashToSourcePath.put(metadata.hash(), path);
                    ++count;
                    continue;
                }
                if (!fileName.endsWith(".gif") || !"cape".equals(type) || (metadata = this.processGifAsset(path)) == null) continue;
                this.metadataCache.put(metadata.hash(), metadata);
                this.hashToSourcePath.put(metadata.hash(), path);
                ++count;
            }
        }
        catch (IOException e) {
            QuickSkin.LOGGER.error("Failed to scan directory: {}", (Object)directory, (Object)e);
        }
        return count;
    }

    private AssetMetadata processPngAsset(Path path, String type) {
        try {
            SkinResolution resolution;
            BufferedImage image;
            int height;
            int width;
            Path metadataPathForCheck;
            String hash = HashUtil.computeFileHash(path);
            if (hash == null) {
                return null;
            }
            if ("cape".equals(type) && !Files.exists(metadataPathForCheck = this.cacheDirectory.resolve(hash + ".json"), new LinkOption[0])) {
                try (InputStream is = Files.newInputStream(path, new OpenOption[0]);){
                    BufferedImage imageForCheck = ImageIO.read(is);
                    if (imageForCheck != null) {
                        int frameCount;
                        width = imageForCheck.getWidth();
                        height = imageForCheck.getHeight();
                        int frameHeight = width / 2;
                        if (width > 0 && frameHeight > 0 && height > frameHeight && height % frameHeight == 0 && (frameCount = height / frameHeight) > 1) {
                            QuickSkin.LOGGER.info("Detected old animated cape strip without metadata: {}. Generating default timing.", (Object)path.getFileName());
                            ArrayList<AnimationMetadata.FrameData> frames = new ArrayList<AnimationMetadata.FrameData>();
                            for (int i = 0; i < frameCount; ++i) {
                                frames.add(new AnimationMetadata.FrameData(50, i));
                            }
                            AnimationMetadata generatedMeta = new AnimationMetadata(frames, frameCount);
                            Files.writeString(metadataPathForCheck, (CharSequence)generatedMeta.toJson(), new OpenOption[0]);
                            QuickSkin.LOGGER.info("Saved default metadata for old cape: {}", (Object)hash);
                        }
                    }
                }
                catch (Exception e) {
                    QuickSkin.LOGGER.warn("Could not check/generate missing metadata for cape '{}': {}", (Object)path.getFileName(), (Object)e.getMessage());
                }
            }
            Path metadataPath = this.cacheDirectory.resolve(hash + ".json");
            AnimationMetadata animMeta = null;
            if (Files.exists(metadataPath, new LinkOption[0])) {
                try {
                    String json = Files.readString(metadataPath);
                    animMeta = AnimationMetadata.fromJson(json);
                }
                catch (IOException e) {
                    QuickSkin.LOGGER.warn("Found metadata file for {} but failed to read it.", (Object)hash, (Object)e);
                }
            }
            if ((image = ImageIO.read(path.toFile())) == null) {
                QuickSkin.LOGGER.warn("Failed to read image: {}", (Object)path);
                return null;
            }
            width = image.getWidth();
            height = image.getHeight();
            String skinModel = null;
            boolean isAnimated = false;
            int frameCount = 1;
            if (animMeta != null) {
                isAnimated = true;
                frameCount = animMeta.frameCount();
                frameHeight = frameCount > 0 ? height / frameCount : height;
                resolution = SkinResolution.fromDimensions(width, frameHeight);
                if (resolution == null) {
                    QuickSkin.LOGGER.warn("Invalid frame dimensions {}x{} for animated asset: {}", new Object[]{width, frameHeight, path});
                    return null;
                }
            } else if ("skin".equals(type)) {
                resolution = SkinResolution.fromDimensions(width, height);
                if (resolution == null) {
                    QuickSkin.LOGGER.warn("Invalid skin dimensions {}x{}: {}", new Object[]{width, height, path});
                    return null;
                }
                skinModel = SkinModelDetector.detectSkinModel(image);
            } else {
                frameHeight = width / 2;
                if (width > 0 && frameHeight > 0 && height % frameHeight == 0) {
                    frameCount = height / frameHeight;
                    isAnimated = frameCount > 1;
                    resolution = SkinResolution.fromDimensions(width, frameHeight);
                    if (resolution == null) {
                        QuickSkin.LOGGER.warn("Invalid cape frame dimensions {}x{}: {}", new Object[]{width, frameHeight, path});
                        return null;
                    }
                } else {
                    QuickSkin.LOGGER.warn("Invalid cape dimensions {}x{}: {}", new Object[]{width, height, path});
                    return null;
                }
            }
            String friendlyName = path.getFileName().toString();
            int dotIndex = friendlyName.lastIndexOf(46);
            if (dotIndex > 0) {
                friendlyName = friendlyName.substring(0, dotIndex);
            }
            long fileSize = Files.size(path);
            if ("skin".equals(type)) {
                return AssetMetadata.forSkin(hash, friendlyName, path, resolution, fileSize, skinModel);
            }
            if (isAnimated) {
                return AssetMetadata.forAnimatedCape(hash, friendlyName, path, resolution, fileSize, frameCount);
            }
            return AssetMetadata.forCape(hash, friendlyName, path, resolution, fileSize);
        }
        catch (Exception e) {
            QuickSkin.LOGGER.error("Failed to process PNG asset: {}", (Object)path, (Object)e);
            return null;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private AssetMetadata processGifAsset(Path path) {
        try (StbGifLoader.GifLoadResult result = null;){
            QuickSkin.LOGGER.info("Processing GIF cape: {}", (Object)path);
            try (InputStream inputStream = Files.newInputStream(path, new OpenOption[0]);){
                result = StbGifLoader.loadGif(inputStream);
            }
            String hash = HashUtil.computeFileHash(path);
            if (hash == null) {
                AssetMetadata assetMetadata = null;
                return assetMetadata;
            }
            int width = result.frameWidth();
            int height = result.frameHeight();
            int frameCount = result.frames().length;
            int atlasHeight = height * frameCount;
            NativeImage atlas = new NativeImage(width, atlasHeight, false);
            for (int i = 0; i < frameCount; ++i) {
                NativeImage frame = result.frames()[i];
                for (int y = 0; y < height; ++y) {
                    for (int x = 0; x < width; ++x) {
                        PlatformHelper.setPixel(atlas, x, i * height + y, PlatformHelper.getPixel(frame, x, y));
                    }
                }
            }
            Path cacheDir = this.cacheDirectory.resolve("animated_capes");
            Files.createDirectories(cacheDir, new FileAttribute[0]);
            Path atlasPath = cacheDir.resolve(hash + ".png");
            atlas.writeToFile(atlasPath);
            atlas.close();
            Path metadataPath = this.cacheDirectory.resolve(hash + ".json");
            Files.writeString(metadataPath, (CharSequence)result.metadata().toJson(), new OpenOption[0]);
            String friendlyName = path.getFileName().toString();
            int dotIndex = friendlyName.lastIndexOf(46);
            if (dotIndex > 0) {
                friendlyName = friendlyName.substring(0, dotIndex);
            }
            long fileSize = Files.size(path);
            SkinResolution resolution = SkinResolution.fromDimensions(width, height);
            if (resolution == null) {
                resolution = SkinResolution.STANDARD;
            }
            QuickSkin.LOGGER.info("GIF cape processed with STBImage: {} frames, hash: {}", (Object)frameCount, (Object)hash);
            AssetMetadata assetMetadata = AssetMetadata.forAnimatedCape(hash, friendlyName, path, resolution, fileSize, frameCount);
            return assetMetadata;
        }
    }

    public AnimationMetadata getAnimationMetadata(String hash) {
        AssetMetadata assetMeta = this.getMetadata(hash);
        if (assetMeta == null || !assetMeta.isAnimated()) {
            return null;
        }
        Path metadataPath = this.cacheDirectory.resolve(hash + ".json");
        if (Files.exists(metadataPath, new LinkOption[0])) {
            try {
                String json = Files.readString(metadataPath);
                return AnimationMetadata.fromJson(json);
            }
            catch (IOException e) {
                QuickSkin.LOGGER.error("Failed to read animation metadata for {}", (Object)hash, (Object)e);
            }
        }
        return null;
    }

    public List<AssetMetadata> getAssetsByType(String type) {
        String playerOwnSkinHash = ClientConfig.getInstance().playerOwnSkinHash;
        return this.metadataCache.values().stream().filter(meta -> type.equals(meta.type())).sorted(Comparator.comparing(meta -> !meta.hash().equals(playerOwnSkinHash)).thenComparing(AssetMetadata::friendlyName)).toList();
    }

    public List<AssetMetadata> getAllSkins() {
        return this.getAssetsByType("skin");
    }

    public AssetMetadata getMetadata(String hash) {
        return this.metadataCache.get(hash);
    }

    public Path getSourcePath(String hash) {
        return this.hashToSourcePath.get(hash);
    }

    public byte[] loadTexture(String hash, TextureQuality quality) {
        Path sourcePath = this.hashToSourcePath.get(hash);
        if (sourcePath == null || !Files.exists(sourcePath, new LinkOption[0])) {
            QuickSkin.LOGGER.debug("Asset not found for hash: {}", (Object)hash);
            return null;
        }
        try {
            boolean shouldRemoveTransparency;
            BufferedImage image = ImageIO.read(sourcePath.toFile());
            if (image == null) {
                return null;
            }
            AssetMetadata metadata = this.getMetadata(hash);
            boolean isSkin = metadata != null && "skin".equals(metadata.type());
            boolean bl = shouldRemoveTransparency = isSkin && ClientConfig.getInstance().shouldDisableSkinTransparency();
            if (shouldRemoveTransparency) {
                image = HDTextureProcessor.removeTransparency(image);
            }
            return switch (quality) {
                default -> throw new MatchException(null, null);
                case TextureQuality.FULL -> {
                    if (shouldRemoveTransparency) {
                        yield HDTextureProcessor.imageToPng(image);
                    }
                    yield Files.readAllBytes(sourcePath);
                }
                case TextureQuality.PREVIEW -> HDTextureProcessor.createPreview(image);
                case TextureQuality.THUMBNAIL -> HDTextureProcessor.createThumbnail(image);
                case TextureQuality.NORMALIZED -> HDTextureProcessor.normalizeForVanilla(image);
            };
        }
        catch (IOException e) {
            QuickSkin.LOGGER.error("Failed to load texture: {}", (Object)hash, (Object)e);
            return null;
        }
    }

    public void deleteAsset(String hash) {
        AssetMetadata metadata = this.metadataCache.get(hash);
        if (metadata == null) {
            return;
        }
        Path path = this.hashToSourcePath.get(hash);
        if (path == null || !Files.exists(path, new LinkOption[0])) {
            return;
        }
        try {
            Files.delete(path);
            this.metadataCache.remove(hash);
            this.hashToSourcePath.remove(hash);
            if (this.skinPreferences != null) {
                this.skinPreferences.remove(hash);
                this.savePreferences();
            }
            QuickSkin.LOGGER.info("Deleted asset: {}", (Object)path);
        }
        catch (IOException e) {
            QuickSkin.LOGGER.error("Failed to delete asset: {}", (Object)path, (Object)e);
        }
    }

    public RenameResult renameLocalAsset(String hash, String newFriendlyName) {
        int dotIndex;
        String extension;
        if (newFriendlyName == null || newFriendlyName.trim().isEmpty()) {
            return RenameResult.INVALID_NAME;
        }
        String sanitizedName = newFriendlyName.trim();
        if (sanitizedName.matches(".*[<>:\"/\\\\|?*].*")) {
            return RenameResult.INVALID_NAME;
        }
        AssetMetadata metadata = this.metadataCache.get(hash);
        if (metadata == null) {
            return RenameResult.NOT_FOUND;
        }
        Path currentPath = this.hashToSourcePath.get(hash);
        if (currentPath == null || !Files.exists(currentPath, new LinkOption[0])) {
            return RenameResult.NOT_FOUND;
        }
        Path parentDir = currentPath.getParent();
        Path newPath = parentDir.resolve(sanitizedName + (extension = (dotIndex = (extension = currentPath.getFileName().toString()).lastIndexOf(46)) > 0 ? extension.substring(dotIndex) : ".png"));
        if (Files.exists(newPath, new LinkOption[0]) && !newPath.equals(currentPath)) {
            return RenameResult.NAME_TAKEN;
        }
        try {
            Files.move(currentPath, newPath, StandardCopyOption.REPLACE_EXISTING);
            AssetMetadata updatedMetadata = "skin".equals(metadata.type()) ? AssetMetadata.forSkin(metadata.hash(), sanitizedName, newPath, metadata.resolution(), metadata.fileSize(), metadata.skinModel()) : (metadata.isAnimated() ? AssetMetadata.forAnimatedCape(metadata.hash(), sanitizedName, newPath, metadata.resolution(), metadata.fileSize(), metadata.frameCount()) : AssetMetadata.forCape(metadata.hash(), sanitizedName, newPath, metadata.resolution(), metadata.fileSize()));
            this.metadataCache.put(hash, updatedMetadata);
            this.hashToSourcePath.put(hash, newPath);
            QuickSkin.LOGGER.info("Renamed asset from '{}' to '{}'", (Object)currentPath, (Object)newPath);
            return RenameResult.SUCCESS;
        }
        catch (IOException e) {
            QuickSkin.LOGGER.error("Failed to rename asset: {}", (Object)currentPath, (Object)e);
            return RenameResult.IO_ERROR;
        }
    }

    public void reload() {
        QuickSkin.LOGGER.info("Reloading local assets...");
        this.discoverLocalAssets();
    }

    public void clearTextureCache() {
        QuickSkin.LOGGER.info("Clearing texture cache (will re-register with current settings)...");
        Minecraft mc = Minecraft.getInstance();
        for (Map<TextureQuality, ResourceLocation> qualityMap : this.textureRegistry.values()) {
            for (ResourceLocation location : qualityMap.values()) {
                try {
                    mc.getTextureManager().release(location);
                }
                catch (Exception e) {
                    QuickSkin.LOGGER.debug("Failed to release texture {}: {}", (Object)location, (Object)e.getMessage());
                }
            }
        }
        this.textureRegistry.clear();
        QuickSkin.LOGGER.info("Texture cache cleared. Textures will reload on next use.");
    }

    public void clearSkinTextureCache() {
        QuickSkin.LOGGER.info("Clearing skin texture cache (will re-register with current settings)...");
        Minecraft mc = Minecraft.getInstance();
        ArrayList<String> hashesToClear = new ArrayList<String>();
        for (String hash : this.textureRegistry.keySet()) {
            AssetMetadata metadata = this.getMetadata(hash);
            if (metadata == null || !"skin".equals(metadata.type())) continue;
            hashesToClear.add(hash);
        }
        for (String hash : hashesToClear) {
            Map<TextureQuality, ResourceLocation> qualityMap = this.textureRegistry.get(hash);
            if (qualityMap == null) continue;
            for (ResourceLocation location : qualityMap.values()) {
                try {
                    mc.getTextureManager().release(location);
                }
                catch (Exception e) {
                    QuickSkin.LOGGER.debug("Failed to release texture {}: {}", (Object)location, (Object)e.getMessage());
                }
            }
            this.textureRegistry.remove(hash);
        }
        QuickSkin.LOGGER.info("Cleared {} skin textures. Capes remain cached.", (Object)hashesToClear.size());
    }

    public ResourceLocation getTextureLocation(String hash, TextureQuality quality) {
        Map qualityMap = this.textureRegistry.get(hash);
        if (qualityMap != null && qualityMap.containsKey((Object)quality)) {
            return qualityMap.get((Object)quality);
        }
        byte[] textureData = this.loadTexture(hash, quality);
        if (textureData == null) {
            return null;
        }
        try {
            BufferedImage bufferedImage = ImageIO.read(new ByteArrayInputStream(textureData));
            if (bufferedImage == null) {
                return null;
            }
            NativeImage nativeImage = this.convertToNativeImage(bufferedImage);
            DynamicTexture dynamicTexture = new DynamicTexture(nativeImage);
            ResourceLocation location = ResourceLocation.fromNamespaceAndPath((String)"quickskin", (String)("local/" + hash + "_" + quality.name().toLowerCase()));
            Minecraft.getInstance().getTextureManager().register(location, (AbstractTexture)dynamicTexture);
            qualityMap = this.textureRegistry.computeIfAbsent(hash, k -> new ConcurrentHashMap());
            qualityMap.put(quality, location);
            QuickSkin.LOGGER.debug("Registered texture: {} ({})", (Object)hash, (Object)quality);
            return location;
        }
        catch (IOException e) {
            QuickSkin.LOGGER.error("Failed to register texture: {}", (Object)hash, (Object)e);
            return null;
        }
    }

    @Nullable
    public BufferedImage getSourceImage(String hash) {
        Path sourcePath = this.getSourcePath(hash);
        if (sourcePath == null) {
            Path cachedAtlas = this.cacheDirectory.resolve("animated_capes").resolve(hash + ".png");
            if (Files.exists(cachedAtlas, new LinkOption[0])) {
                sourcePath = cachedAtlas;
            } else {
                return null;
            }
        }
        try {
            return ImageIO.read(sourcePath.toFile());
        }
        catch (IOException e) {
            QuickSkin.LOGGER.error("Failed to read source image for hash {}: {}", (Object)hash, (Object)e.getMessage());
            return null;
        }
    }

    public Path getSkinsDirectory() {
        return this.skinsDirectory;
    }

    public Path getCapesDirectory() {
        return this.capesDirectory;
    }

    public Path getCacheDirectory() {
        return this.cacheDirectory;
    }

    private NativeImage convertToNativeImage(BufferedImage bufferedImage) {
        int width = bufferedImage.getWidth();
        int height = bufferedImage.getHeight();
        NativeImage nativeImage = new NativeImage(width, height, true);
        for (int y = 0; y < height; ++y) {
            for (int x = 0; x < width; ++x) {
                int argb = bufferedImage.getRGB(x, y);
                int a = argb >> 24 & 0xFF;
                int r = argb >> 16 & 0xFF;
                int g = argb >> 8 & 0xFF;
                int b = argb & 0xFF;
                int abgr = a << 24 | b << 16 | g << 8 | r;
                PlatformHelper.setPixel(nativeImage, x, y, abgr);
            }
        }
        return nativeImage;
    }

    public String getSkinModelPreference(String hash) {
        if (this.skinPreferences == null) {
            QuickSkin.LOGGER.warn("Cannot get model preference - skinPreferences is null, returning default 'auto'");
            return "auto";
        }
        String pref = this.skinPreferences.getModelType(hash);
        QuickSkin.LOGGER.debug("Getting model preference for skin {}: {}", (Object)hash, (Object)pref);
        return pref;
    }

    public void setSkinModelPreference(String hash, String modelType) {
        if (this.skinPreferences != null) {
            QuickSkin.LOGGER.info("Setting model preference for skin {}: {}", (Object)hash, (Object)modelType);
            this.skinPreferences.setModelType(hash, modelType);
            this.savePreferences();
            QuickSkin.LOGGER.info("Saved preferences to: {}", (Object)this.preferencesFile);
        } else {
            QuickSkin.LOGGER.warn("Cannot set model preference - skinPreferences is null");
        }
    }

    private void savePreferences() {
        if (this.skinPreferences != null && this.preferencesFile != null) {
            this.skinPreferences.save(this.preferencesFile);
        }
    }

    public static enum RenameResult {
        SUCCESS,
        NAME_TAKEN,
        INVALID_NAME,
        IO_ERROR,
        NOT_FOUND;

    }
}

