/*
 * Decompiled with CFR 0.152.
 */
package com.quickskin.mod.features.appearance.common;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.mojang.authlib.GameProfile;
import com.mojang.authlib.properties.Property;
import com.quickskin.mod.Config;
import com.quickskin.mod.QuickSkin;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;
import net.minecraft.client.Minecraft;
import net.minecraftforge.api.distmarker.Dist;
import net.minecraftforge.api.distmarker.OnlyIn;
import net.minecraftforge.fml.DistExecutor;

public class SkinAPI {
    private static final Map<String, CachedProfile> profileCache = new ConcurrentHashMap<String, CachedProfile>();
    private static final long CACHE_DURATION_MS = TimeUnit.MINUTES.toMillis(5L);
    private static final Map<String, Long> requestTimestamps = new ConcurrentHashMap<String, Long>();
    private static final long MIN_REQUEST_INTERVAL_MS = 1000L;

    public static CompletableFuture<GameProfile> fetchProfile(String username) {
        long timeSinceLastRequest;
        if (username == null || username.trim().isEmpty()) {
            return CompletableFuture.completedFuture(null);
        }
        String normalizedUsername = username.trim().toLowerCase();
        CachedProfile cached = profileCache.get(normalizedUsername);
        if (cached != null && !cached.isExpired()) {
            return CompletableFuture.completedFuture(cached.profile);
        }
        Long lastRequest = requestTimestamps.get(normalizedUsername);
        if (lastRequest != null && (timeSinceLastRequest = System.currentTimeMillis() - lastRequest) < 1000L) {
            long waitTime = 1000L - timeSinceLastRequest;
            QuickSkin.LOGGER.debug("Rate limiting: waiting {}ms before fetching profile for '{}'", (Object)waitTime, (Object)username);
            return CompletableFuture.supplyAsync(() -> {
                try {
                    Thread.sleep(waitTime);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                }
                return null;
            }).thenCompose(v -> SkinAPI.fetchProfile(username));
        }
        return (CompletableFuture)DistExecutor.unsafeRunForDist(() -> () -> ClientOnly.fetchProfileClient(username, normalizedUsername), () -> () -> SkinAPI.fetchProfileInternal(username));
    }

    private static CompletableFuture<GameProfile> fetchProfileInternal(String username) {
        return CompletableFuture.supplyAsync(() -> {
            String normalizedUsername = username.trim().toLowerCase();
            requestTimestamps.put(normalizedUsername, System.currentTimeMillis());
            HttpURLConnection uuidConnection = null;
            HttpURLConnection profileConnection = null;
            try {
                JsonObject profileJson;
                String name;
                String uuid;
                JsonObject jsonObject;
                URL uuidUrl = new URL("https://api.mojang.com/users/profiles/minecraft/" + username);
                uuidConnection = (HttpURLConnection)uuidUrl.openConnection();
                uuidConnection.setRequestMethod("GET");
                uuidConnection.setConnectTimeout(5000);
                uuidConnection.setReadTimeout(5000);
                uuidConnection.setRequestProperty("User-Agent", "QuickSkin-Mod/1.0");
                int responseCode = uuidConnection.getResponseCode();
                if (responseCode == 404) {
                    QuickSkin.LOGGER.warn("Player '{}' not found", (Object)username);
                    GameProfile gameProfile = null;
                    return gameProfile;
                }
                if (responseCode != 200) {
                    QuickSkin.LOGGER.error("Failed to find player UUID for '{}'. Response code: {}", (Object)username, (Object)responseCode);
                    GameProfile gameProfile = null;
                    return gameProfile;
                }
                try (BufferedReader in = new BufferedReader(new InputStreamReader(uuidConnection.getInputStream()));){
                    jsonObject = JsonParser.parseReader((Reader)in).getAsJsonObject();
                    uuid = jsonObject.get("id").getAsString();
                    name = jsonObject.get("name").getAsString();
                }
                URL profileUrl = new URL("https://sessionserver.mojang.com/session/minecraft/profile/" + uuid + "?unsigned=false");
                profileConnection = (HttpURLConnection)profileUrl.openConnection();
                profileConnection.setRequestMethod("GET");
                profileConnection.setConnectTimeout(5000);
                profileConnection.setReadTimeout(5000);
                profileConnection.setRequestProperty("User-Agent", "QuickSkin-Mod/1.0");
                responseCode = profileConnection.getResponseCode();
                if (responseCode != 200) {
                    QuickSkin.LOGGER.error("Failed to get player profile for UUID '{}'. Response code: {}", (Object)uuid, (Object)responseCode);
                    jsonObject = null;
                    return jsonObject;
                }
                try (BufferedReader profileIn = new BufferedReader(new InputStreamReader(profileConnection.getInputStream()));){
                    profileJson = JsonParser.parseReader((Reader)profileIn).getAsJsonObject();
                }
                String formattedUuid = SkinAPI.formatUuid(uuid);
                GameProfile newProfile = new GameProfile(UUID.fromString(formattedUuid), name);
                if (profileJson.has("properties")) {
                    profileJson.getAsJsonArray("properties").forEach(element -> {
                        JsonObject property = element.getAsJsonObject();
                        String propName = property.get("name").getAsString();
                        String value = property.get("value").getAsString();
                        String signature = property.has("signature") ? property.get("signature").getAsString() : null;
                        newProfile.getProperties().put((Object)propName, (Object)new Property(propName, value, signature));
                    });
                }
                profileCache.put(normalizedUsername, new CachedProfile(newProfile));
                GameProfile gameProfile = newProfile;
                return gameProfile;
            }
            catch (IOException e) {
                QuickSkin.LOGGER.error("Network error while fetching profile for '{}': {}", (Object)username, (Object)e.getMessage());
                GameProfile gameProfile = null;
                return gameProfile;
            }
            catch (Exception e) {
                QuickSkin.LOGGER.error("Unexpected error while fetching profile for '{}'", (Object)username, (Object)e);
                GameProfile gameProfile = null;
                return gameProfile;
            }
            finally {
                if (uuidConnection != null) {
                    uuidConnection.disconnect();
                }
                if (profileConnection != null) {
                    profileConnection.disconnect();
                }
            }
        });
    }

    private static String formatUuid(String uuid) {
        if (uuid.contains("-")) {
            return uuid;
        }
        return uuid.replaceFirst("(\\p{XDigit}{8})(\\p{XDigit}{4})(\\p{XDigit}{4})(\\p{XDigit}{4})(\\p{XDigit}+)", "$1-$2-$3-$4-$5");
    }

    public static void clearCache(String username) {
        if (username != null) {
            String normalizedUsername = username.trim().toLowerCase();
            profileCache.remove(normalizedUsername);
            requestTimestamps.remove(normalizedUsername);
            DistExecutor.unsafeRunWhenOn((Dist)Dist.CLIENT, () -> () -> ClientOnly.clearDiskCacheFor(normalizedUsername));
        }
    }

    public static void clearAllCache() {
        profileCache.clear();
        requestTimestamps.clear();
        DistExecutor.unsafeRunWhenOn((Dist)Dist.CLIENT, () -> () -> ClientOnly.clearAllDiskCache());
    }

    private static class CachedProfile {
        final GameProfile profile;
        final long timestamp;

        CachedProfile(GameProfile profile) {
            this.profile = profile;
            this.timestamp = System.currentTimeMillis();
        }

        boolean isExpired() {
            return System.currentTimeMillis() - this.timestamp > CACHE_DURATION_MS;
        }
    }

    @OnlyIn(value=Dist.CLIENT)
    private static class ClientOnly {
        private static final Gson GSON = new GsonBuilder().setPrettyPrinting().create();
        private static Path CACHE_DIR = null;

        private ClientOnly() {
        }

        private static Path getCacheDir() {
            if (CACHE_DIR == null) {
                CACHE_DIR = Minecraft.m_91087_().f_91069_.toPath().resolve("quickskin_cache").resolve("profiles");
            }
            try {
                if (Files.notExists(CACHE_DIR, new LinkOption[0])) {
                    Files.createDirectories(CACHE_DIR, new FileAttribute[0]);
                }
            }
            catch (IOException e) {
                QuickSkin.LOGGER.error("Failed to create skin cache directory", (Throwable)e);
            }
            return CACHE_DIR;
        }

        private static CompletableFuture<GameProfile> fetchProfileClient(String originalUsername, String normalizedUsername) {
            GameProfile profileFromDisk = ClientOnly.readProfileFromDisk(normalizedUsername);
            if (profileFromDisk != null) {
                return CompletableFuture.completedFuture(profileFromDisk);
            }
            return SkinAPI.fetchProfileInternal(originalUsername).thenApply(networkProfile -> {
                if (networkProfile != null) {
                    ClientOnly.writeProfileToDisk(normalizedUsername, networkProfile);
                }
                return networkProfile;
            });
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        private static GameProfile readProfileFromDisk(String normalizedUsername) {
            if (normalizedUsername.matches(".*[<>:\"/\\\\|?*].*")) {
                QuickSkin.LOGGER.debug("Username '{}' contains invalid path characters, skipping disk cache lookup.", (Object)normalizedUsername);
                return null;
            }
            int durationHours = (Integer)Config.DISK_CACHE_DURATION_HOURS.get();
            if (durationHours == 0) {
                return null;
            }
            Path cacheFile = ClientOnly.getCacheDir().resolve(normalizedUsername + ".json");
            if (Files.notExists(cacheFile, new LinkOption[0])) {
                return null;
            }
            try (BufferedReader reader = Files.newBufferedReader(cacheFile);){
                JsonObject root = JsonParser.parseReader((Reader)reader).getAsJsonObject();
                long cacheTime = root.get("timestamp").getAsLong();
                long maxAge = TimeUnit.HOURS.toMillis(durationHours);
                if (durationHours > 0 && System.currentTimeMillis() - cacheTime > maxAge) {
                    Files.delete(cacheFile);
                    GameProfile gameProfile2 = null;
                    return gameProfile2;
                }
                UUID uuid = UUID.fromString(root.get("uuid").getAsString());
                String name = root.get("name").getAsString();
                GameProfile profile = new GameProfile(uuid, name);
                root.getAsJsonArray("properties").forEach(element -> {
                    JsonObject propJson = element.getAsJsonObject();
                    String propName = propJson.get("name").getAsString();
                    String value = propJson.get("value").getAsString();
                    String signature = propJson.has("signature") ? propJson.get("signature").getAsString() : null;
                    profile.getProperties().put((Object)propName, (Object)new Property(propName, value, signature));
                });
                profileCache.put(normalizedUsername, new CachedProfile(profile));
                GameProfile gameProfile = profile;
                return gameProfile;
            }
            catch (Exception e) {
                QuickSkin.LOGGER.error("Failed to read profile from disk cache for '{}'", (Object)normalizedUsername, (Object)e);
                try {
                    Files.deleteIfExists(cacheFile);
                    return null;
                }
                catch (IOException iOException) {
                    // empty catch block
                }
                return null;
            }
        }

        private static void writeProfileToDisk(String normalizedUsername, GameProfile profile) {
            if ((Integer)Config.DISK_CACHE_DURATION_HOURS.get() == 0) {
                return;
            }
            Path cacheFile = ClientOnly.getCacheDir().resolve(normalizedUsername + ".json");
            JsonObject root = new JsonObject();
            root.addProperty("uuid", profile.getId().toString());
            root.addProperty("name", profile.getName());
            root.addProperty("timestamp", (Number)System.currentTimeMillis());
            JsonArray propertiesArray = new JsonArray();
            profile.getProperties().get((Object)"textures").forEach(p -> {
                JsonObject propJson = new JsonObject();
                propJson.addProperty("name", p.getName());
                propJson.addProperty("value", p.getValue());
                if (p.hasSignature()) {
                    propJson.addProperty("signature", p.getSignature());
                }
                propertiesArray.add((JsonElement)propJson);
            });
            root.add("properties", (JsonElement)propertiesArray);
            try (BufferedWriter writer = Files.newBufferedWriter(cacheFile, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);){
                GSON.toJson((JsonElement)root, (Appendable)writer);
            }
            catch (IOException e) {
                QuickSkin.LOGGER.error("Failed to write profile to disk cache for '{}'", (Object)normalizedUsername, (Object)e);
            }
        }

        private static void clearDiskCacheFor(String normalizedUsername) {
            Path cacheFile = ClientOnly.getCacheDir().resolve(normalizedUsername + ".json");
            try {
                Files.deleteIfExists(cacheFile);
            }
            catch (IOException e) {
                QuickSkin.LOGGER.error("Failed to delete disk cache file for '{}'", (Object)normalizedUsername, (Object)e);
            }
        }

        private static void clearAllDiskCache() {
            Path cacheDir = ClientOnly.getCacheDir();
            if (Files.exists(cacheDir, new LinkOption[0])) {
                try (Stream<Path> files = Files.list(cacheDir);){
                    files.forEach(path -> {
                        try {
                            Files.delete(path);
                        }
                        catch (IOException e) {
                            QuickSkin.LOGGER.error("Failed to delete cached file: {}", path, (Object)e);
                        }
                    });
                }
                catch (IOException e) {
                    QuickSkin.LOGGER.error("Failed to clear disk cache directory", (Throwable)e);
                }
            }
        }
    }
}

