/*
 * Decompiled with CFR 0.152.
 */
package com.quickskin.mod.features.webserver.handlers;

import com.quickskin.mod.Config;
import com.quickskin.mod.QuickSkin;
import com.quickskin.mod.client.ClientSkinManager;
import com.quickskin.mod.core.data.SkinData;
import com.quickskin.mod.core.data.SkinResolution;
import com.quickskin.mod.core.util.SkinModelDetector;
import com.quickskin.mod.features.gui.client.screen.PlayerSkinMenuScreen;
import com.quickskin.mod.features.localstorage.client.LocalAssetManager;
import com.quickskin.mod.features.webserver.dto.ImportRequest;
import com.quickskin.mod.features.webserver.dto.ImportResponse;
import com.quickskin.mod.features.webserver.handlers.BaseHandler;
import com.quickskin.mod.features.webserver.util.JsonHelper;
import com.quickskin.mod.features.webserver.util.RateLimiter;
import com.sun.net.httpserver.HttpExchange;
import com.sun.net.httpserver.HttpHandler;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collection;
import javax.imageio.ImageIO;
import net.minecraft.client.Minecraft;
import net.minecraft.client.resources.sounds.SimpleSoundInstance;
import net.minecraft.client.resources.sounds.SoundInstance;
import net.minecraft.sounds.SoundEvent;
import net.minecraft.sounds.SoundEvents;
import org.slf4j.Logger;

public class ImportSkinHandler
extends BaseHandler
implements HttpHandler {
    private static final Logger LOGGER = QuickSkin.LOGGER;
    private static final int MAX_FILE_SIZE = 0x500000;
    private static final int CONNECTION_TIMEOUT = 10000;
    private static final int READ_TIMEOUT = 15000;
    private final RateLimiter rateLimiter;
    private final String allowedUrlDomain;

    public ImportSkinHandler(String allowedOrigin, RateLimiter rateLimiter, String allowedUrlDomain) {
        super(allowedOrigin);
        this.rateLimiter = rateLimiter;
        this.allowedUrlDomain = allowedUrlDomain;
    }

    @Override
    public void handle(HttpExchange exchange) throws IOException {
        try {
            BufferedImage image;
            byte[] imageData;
            String type;
            String method = exchange.getRequestMethod();
            LOGGER.info("[WebServer] Received {} request to /api/skins/import", (Object)method);
            if (method.equals("OPTIONS")) {
                this.setCorsHeaders(exchange);
                exchange.sendResponseHeaders(204, -1L);
                LOGGER.debug("[WebServer] Handled OPTIONS preflight request");
                return;
            }
            if (!method.equals("POST")) {
                LOGGER.warn("[WebServer] Invalid method: {}", (Object)method);
                this.sendJsonResponse(exchange, 405, new ImportResponse(false, "Method not allowed"));
                return;
            }
            String clientAddress = exchange.getRemoteAddress().getAddress().getHostAddress();
            LOGGER.debug("[WebServer] Checking rate limit for client: {}", (Object)clientAddress);
            if (!this.rateLimiter.allowRequest(clientAddress)) {
                LOGGER.warn("[WebServer] Rate limit exceeded for client: {}", (Object)clientAddress);
                this.sendJsonResponse(exchange, 429, new ImportResponse(false, "Too many requests. Please try again later."));
                return;
            }
            LOGGER.debug("[WebServer] Reading request body...");
            String requestBody = this.readRequestBody(exchange);
            LOGGER.debug("[WebServer] Request body: {}", (Object)requestBody);
            ImportRequest request = JsonHelper.fromJson(requestBody, ImportRequest.class);
            if (request == null) {
                LOGGER.error("[WebServer] Failed to parse request JSON");
                this.sendJsonResponse(exchange, 400, new ImportResponse(false, "Invalid request format"));
                return;
            }
            LOGGER.info("[WebServer] Parsed request - name: '{}', type: '{}', hasUrl: {}, hasBase64: {}", new Object[]{request.getName(), request.getType(), request.getImageUrl() != null, request.getImageBase64() != null});
            if (request.getName() == null || request.getName().trim().isEmpty()) {
                LOGGER.warn("[WebServer] Validation failed: name is required");
                this.sendJsonResponse(exchange, 400, new ImportResponse(false, "Name is required"));
                return;
            }
            if (request.getImageUrl() == null && request.getImageBase64() == null) {
                LOGGER.warn("[WebServer] Validation failed: no image data provided");
                this.sendJsonResponse(exchange, 400, new ImportResponse(false, "Either imageUrl or imageBase64 is required"));
                return;
            }
            String string = type = request.getType() != null ? request.getType().toLowerCase() : "skin";
            if (!type.equals("skin") && !type.equals("cape")) {
                LOGGER.warn("[WebServer] Validation failed: invalid type '{}'", (Object)type);
                this.sendJsonResponse(exchange, 400, new ImportResponse(false, "Type must be either 'skin' or 'cape'"));
                return;
            }
            LOGGER.debug("[WebServer] Validation passed for {} '{}'", (Object)type, (Object)request.getName());
            if (request.getImageUrl() != null && this.allowedUrlDomain != null && !this.allowedUrlDomain.trim().isEmpty()) {
                LOGGER.debug("[WebServer] Checking URL domain restriction for: {}", (Object)request.getImageUrl());
                if (!this.isUrlAllowed(request.getImageUrl())) {
                    LOGGER.warn("[WebServer] URL domain not allowed: {}", (Object)request.getImageUrl());
                    this.sendJsonResponse(exchange, 403, new ImportResponse(false, "URL domain not allowed. Only " + this.allowedUrlDomain + " is permitted."));
                    return;
                }
            }
            try {
                if (request.getImageUrl() != null) {
                    LOGGER.info("[WebServer] Downloading image from: {}", (Object)request.getImageUrl());
                    imageData = this.downloadImage(request.getImageUrl());
                    LOGGER.debug("[WebServer] Downloaded {} bytes", (Object)imageData.length);
                } else {
                    LOGGER.debug("[WebServer] Decoding base64 image data");
                    imageData = Base64.getDecoder().decode(request.getImageBase64());
                    LOGGER.debug("[WebServer] Decoded {} bytes", (Object)imageData.length);
                }
            }
            catch (Exception e) {
                LOGGER.error("[WebServer] Failed to retrieve image data", (Throwable)e);
                this.sendJsonResponse(exchange, 400, new ImportResponse(false, "Failed to retrieve image: " + e.getMessage()));
                return;
            }
            if (imageData.length > 0x500000) {
                LOGGER.warn("[WebServer] Image too large: {} bytes (max: {})", (Object)imageData.length, (Object)0x500000);
                this.sendJsonResponse(exchange, 413, new ImportResponse(false, "Image too large. Max size is 5MB"));
                return;
            }
            LOGGER.debug("[WebServer] Validating image format...");
            try {
                image = ImageIO.read(new ByteArrayInputStream(imageData));
                if (image == null) {
                    LOGGER.error("[WebServer] ImageIO returned null - invalid format");
                    this.sendJsonResponse(exchange, 400, new ImportResponse(false, "Invalid image format. Only PNG and GIF are supported"));
                    return;
                }
                LOGGER.debug("[WebServer] Image valid: {}x{}", (Object)image.getWidth(), (Object)image.getHeight());
            }
            catch (Exception e) {
                LOGGER.error("[WebServer] Failed to parse image", (Throwable)e);
                this.sendJsonResponse(exchange, 400, new ImportResponse(false, "Failed to parse image: " + e.getMessage()));
                return;
            }
            LOGGER.debug("[WebServer] Computing hash...");
            String hash = this.computeHash(imageData);
            LOGGER.info("[WebServer] Computed hash: {}", (Object)hash);
            int width = image.getWidth();
            int height = image.getHeight();
            LOGGER.debug("[WebServer] Determining resolution for {}x{} {}", new Object[]{width, height, type});
            SkinResolution resolution = SkinResolution.fromDimensions(width, height);
            if (resolution != null) {
                LOGGER.debug("[WebServer] Detected resolution: {}", (Object)resolution);
            } else {
                LOGGER.warn("[WebServer] Unknown resolution {}x{}, proceeding without resolution metadata", (Object)width, (Object)height);
            }
            String detectedModel = "default";
            if (type.equals("skin")) {
                detectedModel = SkinModelDetector.detectSkinModel(image);
                LOGGER.info("[WebServer] Auto-detected skin model: {}", (Object)detectedModel);
            }
            try {
                LOGGER.info("[WebServer] Saving texture to upload directory...");
                LocalAssetManager.INSTANCE.saveTextureToUpload(hash, imageData, type, resolution, request.getName());
                LOGGER.info("[WebServer] Successfully imported {} '{}' with hash: {} (resolution: {})", new Object[]{type, request.getName(), hash, resolution});
                if (type.equals("skin")) {
                    String skinId = "local_skin:" + hash;
                    SkinData newSkinData = new SkinData(skinId, detectedModel, "", request.getName());
                    ArrayList<String> savedSkins = new ArrayList<String>((Collection)Config.SAVED_SKINS.get());
                    String encodedSkin = newSkinData.toEncodedString();
                    boolean alreadySaved = savedSkins.stream().anyMatch(s -> SkinData.fromString(s).getUsername().equals(skinId));
                    if (!alreadySaved) {
                        savedSkins.add(encodedSkin);
                        Config.SAVED_SKINS.set(savedSkins);
                        Config.SAVED_SKINS.save();
                        LOGGER.info("[WebServer] Added skin to saved skins config");
                    } else {
                        LOGGER.debug("[WebServer] Skin already in saved skins config");
                    }
                }
                Minecraft mc = Minecraft.m_91087_();
                LOGGER.debug("[WebServer] Minecraft instance obtained, checking player...");
                if (mc.f_91074_ != null) {
                    LOGGER.info("[WebServer] Player exists, applying {} to player UUID: {}", (Object)type, (Object)mc.f_91074_.m_20148_());
                    if (type.equals("skin")) {
                        String skinId = "local_skin:" + hash;
                        LOGGER.debug("[WebServer] Applying skin with ID: {}", (Object)skinId);
                        ClientSkinManager.getInstance().applyLook(mc.f_91074_.m_20148_(), skinId, null, null);
                        LOGGER.info("[WebServer] Applied skin to player");
                    } else {
                        String capeId = "local_cape:" + hash;
                        LOGGER.debug("[WebServer] Applying cape with ID: {}", (Object)capeId);
                        ClientSkinManager.getInstance().applyLook(mc.f_91074_.m_20148_(), mc.f_91074_.m_7755_().getString(), capeId, null);
                        LOGGER.info("[WebServer] Applied cape to player");
                    }
                } else {
                    LOGGER.warn("[WebServer] Player is null, skin saved but not applied (not in world)");
                }
                LOGGER.debug("[WebServer] Scheduling UI refresh on main thread...");
                String importedSkinId = type.equals("skin") ? "local_skin:" + hash : null;
                mc.execute(() -> {
                    LOGGER.debug("[WebServer] UI refresh task executing on main thread");
                    LOGGER.debug("[WebServer] Current screen: {}", (Object)(mc.f_91080_ != null ? mc.f_91080_.getClass().getSimpleName() : "null"));
                    mc.m_91106_().m_120367_((SoundInstance)SimpleSoundInstance.m_119755_((SoundEvent)SoundEvents.f_12275_, (float)1.5f, (float)0.5f));
                    LOGGER.debug("[WebServer] Played success sound for import");
                    if (mc.f_91080_ instanceof PlayerSkinMenuScreen) {
                        LOGGER.info("[WebServer] PlayerSkinMenuScreen is open, calling refreshSkinList()");
                        PlayerSkinMenuScreen skinMenu = (PlayerSkinMenuScreen)mc.f_91080_;
                        skinMenu.refreshSkinList();
                        if (importedSkinId != null) {
                            skinMenu.selectSkinInList(importedSkinId);
                            LOGGER.info("[WebServer] Skin list refreshed and imported skin selected: {}", (Object)importedSkinId);
                        } else {
                            LOGGER.info("[WebServer] Skin list refreshed (cape import, no selection)");
                        }
                    } else {
                        LOGGER.info("[WebServer] PlayerSkinMenuScreen is not open, skipping refresh");
                    }
                });
                ImportResponse response = new ImportResponse(true, "Successfully imported " + type, hash, type);
                LOGGER.debug("[WebServer] Sending success response");
                this.sendJsonResponse(exchange, 200, response);
                LOGGER.info("[WebServer] Import request completed successfully");
            }
            catch (Exception e) {
                LOGGER.error("[WebServer] Failed to save texture", (Throwable)e);
                this.sendJsonResponse(exchange, 500, new ImportResponse(false, "Failed to save texture: " + e.getMessage()));
            }
        }
        catch (Exception e) {
            LOGGER.error("[WebServer] Error handling import request", (Throwable)e);
            try {
                this.sendJsonResponse(exchange, 500, new ImportResponse(false, "Internal server error"));
            }
            catch (IOException ioException) {
                LOGGER.error("[WebServer] Failed to send error response", (Throwable)ioException);
            }
        }
    }

    private byte[] downloadImage(String imageUrl) throws IOException {
        if (imageUrl.startsWith("data:")) {
            LOGGER.debug("[WebServer] Processing data URI");
            int commaIndex = imageUrl.indexOf(44);
            if (commaIndex == -1) {
                throw new IOException("Invalid data URI format");
            }
            String base64Data = imageUrl.substring(commaIndex + 1);
            return Base64.getDecoder().decode(base64Data);
        }
        URL url = new URL(imageUrl);
        HttpURLConnection connection = (HttpURLConnection)url.openConnection();
        connection.setRequestMethod("GET");
        connection.setConnectTimeout(10000);
        connection.setReadTimeout(15000);
        connection.setRequestProperty("User-Agent", "QuickSkin/2.4.1");
        int responseCode = connection.getResponseCode();
        if (responseCode != 200) {
            throw new IOException("Failed to download image. HTTP " + responseCode);
        }
        try (InputStream in = connection.getInputStream();){
            byte[] byArray = in.readAllBytes();
            return byArray;
        }
    }

    private String computeHash(byte[] data) {
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA-1");
            byte[] hash = digest.digest(data);
            StringBuilder hexString = new StringBuilder();
            for (byte b : hash) {
                String hex = Integer.toHexString(0xFF & b);
                if (hex.length() == 1) {
                    hexString.append('0');
                }
                hexString.append(hex);
            }
            return hexString.toString();
        }
        catch (Exception e) {
            LOGGER.error("Failed to compute hash", (Throwable)e);
            return String.valueOf(System.currentTimeMillis());
        }
    }

    private boolean isUrlAllowed(String urlString) {
        try {
            URL url = new URL(urlString);
            String host = url.getHost().toLowerCase();
            String allowedDomain = this.allowedUrlDomain.toLowerCase().trim();
            if (host.equals(allowedDomain)) {
                return true;
            }
            return host.endsWith("." + allowedDomain);
        }
        catch (Exception e) {
            LOGGER.error("Invalid URL format: {}", (Object)urlString, (Object)e);
            return false;
        }
    }
}

