/*
 * Decompiled with CFR 0.152.
 */
package com.quickskin.mod.client;

import com.mojang.blaze3d.platform.NativeImage;
import com.quickskin.mod.Config;
import com.quickskin.mod.QuickSkin;
import com.quickskin.mod.client.LocalSkinManager;
import com.quickskin.mod.util.GifUtil;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import net.minecraft.client.Minecraft;
import net.minecraft.client.renderer.texture.DynamicTexture;
import net.minecraft.resources.ResourceLocation;
import net.minecraftforge.api.distmarker.Dist;
import net.minecraftforge.api.distmarker.OnlyIn;

@OnlyIn(value=Dist.CLIENT)
public class AnimatedTextureManager {
    private static final Map<String, AnimationState> animations = new ConcurrentHashMap<String, AnimationState>();

    public static AnimationState registerAnimation(String textureId, NativeImage atlas, List<GifUtil.FrameData> frames, int frameCount) {
        int maxConcurrent = (Integer)Config.MAX_CONCURRENT_ANIMATIONS.get();
        if (animations.size() >= maxConcurrent && !animations.containsKey(textureId)) {
            QuickSkin.LOGGER.warn("Maximum concurrent animations ({}) reached. Evicting oldest animation.", (Object)maxConcurrent);
            String oldestId = null;
            long oldestTime = Long.MAX_VALUE;
            for (Map.Entry<String, AnimationState> entry : animations.entrySet()) {
                if (entry.getValue().getLastAccessTime() >= oldestTime) continue;
                oldestTime = entry.getValue().getLastAccessTime();
                oldestId = entry.getKey();
            }
            if (oldestId != null) {
                QuickSkin.LOGGER.info("Evicting animation: {}", oldestId);
                AnimatedTextureManager.removeAnimation(oldestId);
            } else {
                QuickSkin.LOGGER.error("Could not find an animation to evict. Registration of {} failed.", (Object)textureId);
                return null;
            }
        }
        AnimatedTextureManager.removeAnimation(textureId);
        AnimationState state = new AnimationState(textureId, atlas, frames, frameCount);
        if (!state.isValid()) {
            QuickSkin.LOGGER.error("Failed to register animation: {}", (Object)textureId);
            return null;
        }
        animations.put(textureId, state);
        return state;
    }

    public static AnimationState getAnimation(String textureId) {
        return animations.get(textureId);
    }

    public static boolean hasAnimation(String textureId) {
        AnimationState state = animations.get(textureId);
        return state != null && state.isValid();
    }

    public static ResourceLocation getCurrentFrameTexture(String textureId) {
        AnimationState state = animations.get(textureId);
        return state != null && state.isValid() ? state.getCurrentTexture() : null;
    }

    public static void tickAll() {
        for (String textureId : animations.keySet().toArray(new String[0])) {
            AnimationState state = animations.get(textureId);
            if (state == null) continue;
            if (state.isValid()) {
                state.tick();
                continue;
            }
            AnimatedTextureManager.removeAnimation(textureId);
        }
        if (Minecraft.m_91087_().f_91065_.m_93079_() % 1200 == 0) {
            AnimatedTextureManager.checkMemoryUsage();
        }
    }

    public static void updateInterpolation(float renderTickTime) {
    }

    public static void removeAnimation(String textureId) {
        AnimationState state = animations.remove(textureId);
        if (state != null) {
            state.cleanup();
            QuickSkin.LOGGER.debug("Removed animation: {}", (Object)textureId);
        }
    }

    public static void cleanup() {
        QuickSkin.LOGGER.info("Cleaning up {} animations", (Object)animations.size());
        animations.values().forEach(AnimationState::cleanup);
        animations.clear();
    }

    public static int getActiveCount() {
        return animations.size();
    }

    public static void debugAnimationStatus(String textureId) {
        AnimationState state = animations.get(textureId);
        if (state != null) {
            QuickSkin.LOGGER.info("Animation {} - Current frame: {}/{}, Tick counter: {}, Valid: {}", new Object[]{textureId, state.getCurrentFrame(), state.getFrameCount(), state.getTickCounter(), state.isValid()});
        } else {
            QuickSkin.LOGGER.info("No animation found for: {}", (Object)textureId);
        }
    }

    public static long getEstimatedMemoryUsage() {
        return animations.values().stream().mapToLong(AnimationState::getMemoryUsage).sum();
    }

    private static void checkMemoryUsage() {
        long memoryLimitBytes;
        long memoryUsage = AnimatedTextureManager.getEstimatedMemoryUsage();
        if (memoryUsage > (memoryLimitBytes = (long)((Integer)Config.ANIMATION_MEMORY_LIMIT_MB.get()).intValue() * 1024L * 1024L)) {
            QuickSkin.LOGGER.warn("Animation memory usage ({} MB) exceeds limit ({} MB), cleaning up all animations", (Object)(memoryUsage / 1024L / 1024L), (Object)(memoryLimitBytes / 1024L / 1024L));
            AnimatedTextureManager.cleanup();
        }
    }

    private static String sanitizeTextureId(String textureId) {
        if (textureId == null) {
            return "animated_cape";
        }
        return textureId.toLowerCase().replaceAll("[^a-z0-9_.-]", "_").replaceAll("_{2,}", "_");
    }

    public static String[] getAllAnimationIds() {
        return animations.keySet().toArray(new String[0]);
    }

    public static boolean refreshAnimation(String textureId) {
        NativeImage nativeAtlas;
        String hash = textureId.replace("cape_", "");
        if (hash.isEmpty()) {
            return false;
        }
        LocalSkinManager.AnimationMetadata meta = LocalSkinManager.INSTANCE.getAnimationMetadata(hash);
        BufferedImage sourceAtlasImage = LocalSkinManager.INSTANCE.getSourceImage(hash);
        if (meta != null && sourceAtlasImage != null && (nativeAtlas = LocalSkinManager.convertToNativeImage(sourceAtlasImage)) != null) {
            AnimationState newState = AnimatedTextureManager.registerAnimation(textureId, nativeAtlas, meta.frames(), meta.frameCount());
            return newState != null && newState.isValid();
        }
        return false;
    }

    public static class AnimationState {
        private final String textureId;
        private final List<GifUtil.FrameData> frames;
        private final DynamicTexture[] frameTextures;
        private final ResourceLocation[] frameResourceLocations;
        private int currentFrame = 0;
        private int tickCounter = 0;
        private boolean isValid;
        private final long atlasMemoryUsage;
        private long lastAccessTime;

        public AnimationState(String textureId, NativeImage atlas, List<GifUtil.FrameData> frames, int frameCount) {
            long calculatedMemoryUsage;
            this.textureId = textureId;
            this.frames = frames;
            this.lastAccessTime = System.currentTimeMillis();
            if (atlas == null || frames == null || frames.isEmpty() || frameCount <= 1) {
                QuickSkin.LOGGER.warn("Invalid animation data for: {}", (Object)textureId);
                this.frameTextures = new DynamicTexture[0];
                this.frameResourceLocations = new ResourceLocation[0];
                this.isValid = false;
                this.atlasMemoryUsage = 0L;
                return;
            }
            int maxFrames = (Integer)Config.MAX_GIF_FRAMES.get();
            int actualFrames = Math.min(frameCount, maxFrames);
            if (actualFrames < frameCount) {
                QuickSkin.LOGGER.warn("Animation {} has {} frames, limiting to {}", new Object[]{textureId, frameCount, actualFrames});
            }
            this.frameTextures = new DynamicTexture[actualFrames];
            this.frameResourceLocations = new ResourceLocation[actualFrames];
            try {
                long tempMemoryUsage = (long)atlas.m_84982_() * (long)atlas.m_85084_() * 4L;
                for (int i = 0; i < actualFrames; ++i) {
                    DynamicTexture dynamicTexture;
                    NativeImage frameImage = this.extractFrame(atlas, i, frameCount);
                    if (frameImage == null) {
                        throw new IOException("Failed to extract frame " + i);
                    }
                    this.frameTextures[i] = dynamicTexture = new DynamicTexture(frameImage);
                    String sanitizedId = AnimatedTextureManager.sanitizeTextureId(textureId) + "_frame_" + i;
                    this.frameResourceLocations[i] = Minecraft.m_91087_().m_91097_().m_118490_(sanitizedId, dynamicTexture);
                }
                this.isValid = true;
                calculatedMemoryUsage = tempMemoryUsage;
            }
            catch (Exception e) {
                QuickSkin.LOGGER.error("Failed to create animation for {}", (Object)textureId, (Object)e);
                this.isValid = false;
                calculatedMemoryUsage = 0L;
            }
            this.atlasMemoryUsage = calculatedMemoryUsage;
            if (!this.isValid) {
                this.cleanup();
            }
        }

        private NativeImage extractFrame(NativeImage atlas, int frameIndex, int totalFrames) {
            try {
                int frameHeight = atlas.m_85084_() / totalFrames;
                int yOffset = frameIndex * frameHeight;
                NativeImage frameImage = new NativeImage(atlas.m_84982_(), frameHeight, true);
                atlas.m_85034_(0, yOffset, atlas.m_84982_(), frameHeight, frameImage);
                return frameImage;
            }
            catch (Exception e) {
                QuickSkin.LOGGER.error("Failed to extract frame {} from NativeImage atlas", (Object)frameIndex, (Object)e);
                return null;
            }
        }

        public void tick() {
            int finalDelay;
            int frameTicks;
            if (!this.isValid || this.frames.isEmpty() || this.frameResourceLocations.length == 0) {
                return;
            }
            ++this.tickCounter;
            int currentFrameDelay = this.frames.get(this.currentFrame).delay();
            int maxFps = (Integer)Config.MAX_ANIMATION_FPS.get();
            int minDelay = 1000 / maxFps;
            currentFrameDelay = Math.max(currentFrameDelay, minDelay);
            double speedMultiplier = (Double)Config.ANIMATION_SPEED_MULTIPLIER.get();
            if (speedMultiplier <= 0.05) {
                speedMultiplier = 0.05;
            }
            if (this.tickCounter >= (frameTicks = Math.max(1, (finalDelay = (int)((double)currentFrameDelay / speedMultiplier)) / 50))) {
                this.currentFrame = (this.currentFrame + 1) % this.frameResourceLocations.length;
                this.tickCounter = 0;
            }
        }

        public ResourceLocation getCurrentTexture() {
            if (!this.isValid || this.frameResourceLocations.length == 0) {
                return null;
            }
            this.lastAccessTime = System.currentTimeMillis();
            return this.frameResourceLocations[this.currentFrame];
        }

        public boolean isValid() {
            return this.isValid;
        }

        public int getCurrentFrame() {
            return this.currentFrame;
        }

        public int getFrameCount() {
            return this.frameResourceLocations.length;
        }

        public int getTickCounter() {
            return this.tickCounter;
        }

        public long getMemoryUsage() {
            return this.atlasMemoryUsage;
        }

        public long getLastAccessTime() {
            return this.lastAccessTime;
        }

        public void cleanup() {
            for (ResourceLocation resourceLocation : this.frameResourceLocations) {
                if (resourceLocation == null) continue;
                Minecraft.m_91087_().m_91097_().m_118513_(resourceLocation);
            }
            for (ResourceLocation resourceLocation : this.frameTextures) {
                if (resourceLocation == null) continue;
                resourceLocation.close();
            }
            QuickSkin.LOGGER.debug("Cleaned up animation: {}", (Object)this.textureId);
        }
    }
}

