package qouteall.imm_ptl.peripheral.alternate_dimension;

import it.unimi.dsi.fastutil.doubles.DoubleList;

import java.util.Random;
import java.util.function.Function;

public class FormulaGenerator {
    
    public static interface TriNumFunction {
        double eval(double x, double y, double z);
        
        default TriNumFunction normalize(
        ) {
            DoubleList doubleList = DoubleList.of(
                eval(0, 0, 0),
                eval(0, 0, 1),
                eval(0, 1, 0),
                eval(0, 1, 1),
                eval(1, 0, 0),
                eval(1, 0, 1),
                eval(1, 1, 0),
                eval(1, 1, 1)
            );
            double mean = doubleList.doubleStream().sum() / 8;
            double S = doubleList.doubleStream().map(v -> (v - mean) * (v - mean)).sum() / 8;
            double deviation = Math.sqrt(S);
            
            return (x, y, z) -> (eval(x, y, z) - mean) / deviation;
        }
    }
    
    public static interface UniNumFunction {
        double eval(double x);
    }
    
    // input 3 nums, output 3 nums.
    // to avoid garbage generated by Vec3d, use continuation passing style
    public static interface TriToTriFuncCPS {
        double eval(double x, double y, double z, TriNumFunction continuation);
        
        // CPS combination
        default TriToTriFuncCPS then(TriToTriFuncCPS other) {
            return (x, y, z, c) -> this.eval(x, y, z, (nx, ny, nz) -> other.eval(nx, ny, nz, c));
        }
        
        default TriNumFunction finish(TriNumFunction finalCont) {
            return (x, y, z) -> this.eval(x, y, z, finalCont);
        }
    }
    
    private static RandomSelector<Function<Random, UniNumFunction>> uniFuncSelector;
    private static RandomSelector<TriNumFunction> triFuncSelector;
    
    public static void init() {
        initUniFuncSelector();
        initTriFuncSelector();
    }
    
    private static void initUniFuncSelector() {
        RandomSelector.Builder<Function<Random, UniNumFunction>> builder = new RandomSelector.Builder<>();
        
        Function<Function<Double, UniNumFunction>, Function<Random, UniNumFunction>> applyRandomArg =
            arged -> random -> arged.apply(random.nextDouble());
        Function<UniNumFunction, Function<Random, UniNumFunction>> noRandomArg =
            func -> random -> func;
        
        builder.add(30, noRandomArg.apply(x -> x));
        builder.add(30, noRandomArg.apply(x -> -x));
        builder.add(30, noRandomArg.apply(x -> x * x));
        builder.add(30, noRandomArg.apply(x -> x * x * x));
        builder.add(10, noRandomArg.apply(x -> Math.max(x, 0)));
        builder.add(7, noRandomArg.apply(x -> Math.sin(x * 3)));
        builder.add(7, noRandomArg.apply(x -> Math.cos(x * 7)));
        builder.add(10, noRandomArg.apply(x -> Math.exp(x * 3)));
        builder.add(10, noRandomArg.apply(x -> Math.log(Math.abs(x) + 1)));
        builder.add(5, noRandomArg.apply(x -> Math.cosh(x)));
        builder.add(10, noRandomArg.apply(x -> Math.abs(x)));
        builder.add(5, noRandomArg.apply(x -> Math.round(x)));
        builder.add(10, noRandomArg.apply(x -> Math.sqrt(Math.abs(x))));
        builder.add(10, noRandomArg.apply(x -> weirdSwap(x)));
        builder.add(30, applyRandomArg.apply(arg -> x -> x * arg));
        builder.add(30, applyRandomArg.apply(arg -> x -> x / Math.max(arg, 0.1)));
        builder.add(10, applyRandomArg.apply(arg -> x -> Math.max(x, arg)));
        builder.add(10, applyRandomArg.apply(arg -> x -> x + arg));
        builder.add(7, applyRandomArg.apply(arg -> x -> Math.floor(x * arg * 23)));
        builder.add(7, applyRandomArg.apply(arg -> x -> Math.floor(x * arg)));
        builder.add(10, applyRandomArg.apply(arg -> x -> weirdAnd(arg, x)));
        builder.add(10, applyRandomArg.apply(arg -> x -> weirdXor(arg, x)));
        
        uniFuncSelector = builder.build();
    }
    
    private static int toInt(double d) {
        return ((int) (d * 256) + 256);
    }
    
    private static double fromInt(int d) {
        return (double) (((double) d - 256) / 256);
    }
    
    private static double weirdAnd(double a, double b) {
        return fromInt(toInt(a) & toInt(b));
    }
    
    private static double weirdXor(double a, double b) {
        return fromInt(toInt(a) ^ toInt(b));
    }
    
    private static double weirdSwap(double a) {
        int i = toInt(a);
        
        int r = ((i & 0x0F) << 4) | ((i & 0xF0) >> 4);
        
        return fromInt(r);
    }
    
    private static void initTriFuncSelector() {
        RandomSelector.Builder<TriNumFunction> builder = new RandomSelector.Builder<>();
        
        builder.add(50, (x, y, z) -> x + y + z);
        builder.add(50, (x, y, z) -> x * y * z);
        builder.add(10, (x, y, z) -> x * y + z);
        builder.add(10, (x, y, z) -> x + y * z);
        builder.add(10, (x, y, z) -> x * z + y);
        builder.add(10, (x, y, z) -> x + y + z);
        builder.add(10, (x, y, z) -> x * x + y * y + z * z);
        builder.add(10, (x, y, z) -> x + y * y + z * z);
        builder.add(10, (x, y, z) -> x * x + y + z * z);
        builder.add(10, (x, y, z) -> x * x + y * y + z);
        builder.add(10, (x, y, z) -> -x * x + y * y + z * z);
        builder.add(10, (x, y, z) -> x * x - y * y + z * z);
        builder.add(10, (x, y, z) -> x * x + y * y - z * z);
        builder.add(10, (x, y, z) -> (x + y + z) * (x + y + z));
        builder.add(10, (x, y, z) -> (x + y + z) * (x + y - z));
        builder.add(10, (x, y, z) -> x * x * x * y * z);
        builder.add(5, (x, y, z) -> Math.pow(y, x + z));
        builder.add(5, (x, y, z) -> Math.pow(y, x - z));
        builder.add(5, (x, y, z) -> Math.pow(x * z, y));
        builder.add(10, (x, y, z) -> Math.log(Math.abs(x + y + z) + 0.5));
        builder.add(10, (x, y, z) -> Math.log(Math.abs(x - y * z) + 0.5));
        
        triFuncSelector = builder.build();
    }
    
    public static TriNumFunction newGetRandomTriCompositeExpression(Random random, int nestLayer) {
        if (nestLayer == 0) {
            return shuffleFunSelector.select(random).finish(triFuncSelector.select(random));
        }
        else {
            TriToTriFuncCPS curr = newGetComplexTriNumFunc2(random);
            
            curr = curr.then(newGetComplexTriNumFunc1(random));
            
            return curr.finish(
                newGetRandomTriCompositeExpression(random, nestLayer - 1)
            );
        }
    }
    
    public static TriToTriFuncCPS newGetComplexTriNumFunc1(Random random) {
        TriNumFunction f1 = triFuncSelector.select(random);
        TriNumFunction f2 = triFuncSelector.select(random);
        TriNumFunction f3 = triFuncSelector.select(random);
        
        return (x, y, z, c) -> c.eval(
            f1.eval(x, y, z), f2.eval(x, y, z), f3.eval(x, y, z)
        );
    }
    
    public static TriToTriFuncCPS newGetComplexTriNumFunc2(Random random) {
        UniNumFunction f1 = uniFuncSelector.select(random).apply(random);
        UniNumFunction f2 = uniFuncSelector.select(random).apply(random);
        UniNumFunction f3 = uniFuncSelector.select(random).apply(random);
        
        return (x, y, z, c) -> c.eval(
            f1.eval(x), f2.eval(y), f3.eval(z)
        );
    }
    
    private static final RandomSelector<TriToTriFuncCPS> shuffleFunSelector =
        new RandomSelector.Builder<TriToTriFuncCPS>()
            .add(1, (x, y, z, c) -> (c.eval(x, y, z)))
            .add(1, (x, y, z, c) -> (c.eval(x, z, y)))
            .add(1, (x, y, z, c) -> (c.eval(y, z, x)))
            .add(1, (x, y, z, c) -> (c.eval(y, x, z)))
            .add(1, (x, y, z, c) -> (c.eval(z, x, y)))
            .add(1, (x, y, z, c) -> (c.eval(z, y, x)))
            .build();
    
    
}
