package qouteall.imm_ptl.peripheral.wand;

import com.mojang.logging.LogUtils;
import net.minecraft.class_243;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import qouteall.imm_ptl.core.portal.Portal;
import qouteall.imm_ptl.core.portal.animation.UnilateralPortalState;
import qouteall.q_misc_util.my_util.Circle;
import qouteall.q_misc_util.my_util.DQuaternion;
import qouteall.q_misc_util.my_util.Plane;
import qouteall.q_misc_util.my_util.Sphere;

public enum PortalCorner {
    LEFT_BOTTOM, LEFT_TOP, RIGHT_BOTTOM, RIGHT_TOP;
    
    private static final Logger LOGGER = LogUtils.getLogger();
    
    public int getXSign() {
        return switch (this) {
            case LEFT_BOTTOM, LEFT_TOP -> -1;
            case RIGHT_BOTTOM, RIGHT_TOP -> 1;
        };
    }
    
    public int getYSign() {
        return switch (this) {
            case LEFT_BOTTOM, RIGHT_BOTTOM -> -1;
            case LEFT_TOP, RIGHT_TOP -> 1;
        };
    }
    
    public class_243 getOffset(Portal portal) {
        return portal.axisW.method_1021((portal.width / 2) * getXSign())
            .method_1019(portal.axisH.method_1021((portal.height / 2) * getYSign()));
    }
    
    public class_243 getPos(Portal portal) {
        return portal.getOriginPos().method_1019(getOffset(portal));
    }
    
    public class_243 getOffset(UnilateralPortalState ups) {
        return ups.getAxisW().method_1021((ups.width() / 2) * getXSign())
            .method_1019(ups.getAxisH().method_1021((ups.height() / 2) * getYSign()));
    }
    
    public class_243 getPos(UnilateralPortalState ups) {
        return ups.position().method_1019(getOffset(ups));
    }
    
    public static UnilateralPortalState performDragWithNoLockedCorner(
        UnilateralPortalState originalState,
        PortalCorner freeCorner, class_243 freePos
    ) {
        // simply move the portal
        class_243 offset = freeCorner.getOffset(originalState);
        class_243 newPos = freePos.method_1020(offset);
        
        return new UnilateralPortalState.Builder()
            .from(originalState)
            .position(newPos)
            .build();
    }
    
    @Nullable
    public static UnilateralPortalState performDragWith1LockedCorner(
        UnilateralPortalState originalState,
        PortalCorner lockedCorner, class_243 lockedPos,
        PortalCorner freeCorner, class_243 freePos
    ) {
        // rotate the portal along the locked corner and do scaling
        
        class_243 originalFreeCornerPos = freeCorner.getPos(originalState);
        
        class_243 originalOffset = originalFreeCornerPos.method_1020(lockedPos);
        class_243 newOffset = freePos.method_1020(lockedPos);
        
        if (originalOffset.method_1027() < 0.001 || newOffset.method_1027() < 0.001) {
            return null;
        }
        
        double dot = originalOffset.method_1029().method_1026(newOffset.method_1029());
        
        DQuaternion rotation;
        if (Math.abs(dot) > 0.99999) {
            // the rotation cannot be determined if the two vecs are parallel
            rotation = null;
        }
        else {
            rotation = DQuaternion.getRotationBetween(originalOffset, newOffset);
        }
        
        double scaling = newOffset.method_1033() / originalOffset.method_1033();
        
        class_243 offset = originalState.position().method_1020(lockedPos).method_1021(scaling);
        
        if (rotation != null) {
            offset = rotation.rotate(offset);
        }
        
        class_243 newOrigin = lockedPos.method_1019(offset);
        
        DQuaternion newOrientation = rotation == null ? originalState.orientation() :
            rotation.hamiltonProduct(originalState.orientation());
        
        double newWidth = originalState.width() * scaling;
        double newHeight = originalState.height() * scaling;
        
        return new UnilateralPortalState.Builder()
            .from(originalState)
            .position(newOrigin)
            .orientation(newOrientation)
            .width(newWidth)
            .height(newHeight)
            .build();
    }
    
    public static record DraggingConstraint(
        @Nullable Plane plane,
        @Nullable Sphere sphere
    ) {
        @Nullable
        public class_243 constrain(class_243 pos) {
            if (sphere != null) {
                if (plane != null) {
                    Circle circle = sphere.getIntersectionWithPlane(plane);
                    
                    if (circle == null) {
                        return null;
                    }
                    
                    return circle.projectToCircle(pos);
                }
                else {
                    // limit on sphere
                    return sphere.projectToSphere(pos);
                }
            }
            else {
                if (plane != null) {
                    // limit on plane
                    return plane.getProjection(pos);
                }
                else {
                    // no limit
                    return pos;
                }
            }
        }
    }
    
    /**
     * two cases
     * 1. the two locked corners are on one edge
     * 2. the two locked corners are diagonal
     * <p>
     * in case 1
     * (locked1 - locked2) (locked2 - free) = 0
     * the free point is in a plane
     * <p>
     * in case 2
     * (locked1 - free) (locked2 - free) = 0
     * let center = (locked1 + locked2) / 2
     * locked1 = center + (locked1 - center)
     * => (center - free + (locked1 - center)) (center - free - (locked1 - center)) = 0
     * => (center - free)^2 = (locked1 - center)^2
     * the free point is in a sphere
     * <p>
     * if the aspect ratio is locked, |free - locked1| = k |free - locked2|
     * => (free - locked1)^2 = k^2 (free - locked2)^2
     * => (free - center - (locked1 - center))^2 = k^2 (free - center + (locked1 - center))
     * => 2 (locked1 - center)^2 - 2 (free - center) (locked1 - center) =
     * k^2 ( 2 (locked1 - center)^2 + 2 (free - center) (locked1 - center) )
     * => (1 - k^2) (locked1 - center)^2 = (1 + k^2) (free - center) (locked1 - center)
     * => (free - center) (locked1 - center) = ((1 - k^2) / (1 + k^2)) (locked1 - center)^2
     * the free point is also limited in a plane
     */
    public static DraggingConstraint getDraggingConstraintWith2LockedCorners(
        PortalCorner lockedCorner1, class_243 lockedPos1,
        PortalCorner lockedCorner2, class_243 lockedPos2,
        PortalCorner freeCorner
    ) {
        int corner1XSign = lockedCorner1.getXSign();
        int corner1YSign = lockedCorner1.getYSign();
        int corner2XSign = lockedCorner2.getXSign();
        int corner2YSign = lockedCorner2.getYSign();
        int freeCornerXSign = freeCorner.getXSign();
        int freeCornerYSign = freeCorner.getYSign();
        
        if (corner1XSign == corner2XSign || corner1YSign == corner2YSign) {
            // case 1
            // plane: (locked1 - locked2) (locked2 - free) = 0
            
            if (freeCornerXSign == corner1XSign || freeCornerYSign == corner1YSign) {
                // the free corner is close to the lockedCorner1
                class_243 planeNormal = lockedPos2.method_1020(lockedPos1);
                Plane plane = new Plane(lockedPos1, planeNormal);
                
                return new DraggingConstraint(
                    plane, null
                );
            }
            else {
                // the free corner is close the lockedCorner2
                class_243 planeNormal = lockedPos1.method_1020(lockedPos2);
                Plane plane = new Plane(lockedPos2, planeNormal);
                
                return new DraggingConstraint(
                    plane, null
                );
            }
        }
        else {
            // case 2
            // sphere: (free - center)^2 = (locked1 - center)^2
            class_243 center = lockedPos1.method_1019(lockedPos2).method_1021(0.5);
            double radius = lockedPos1.method_1022(lockedPos2) * 0.5;
            
            return new DraggingConstraint(
                null,
                new Sphere(center, radius)
            );
            
            // aspect ratio lock not implemented yet
        }
    }
    
    @Nullable
    public static UnilateralPortalState performDragWith2LockedCorners(
        UnilateralPortalState originalState,
        PortalCorner lockedCorner1, class_243 lockedPos1,
        PortalCorner lockedCorner2, class_243 lockedPos2,
        PortalCorner freeCorner, class_243 freePos
    ) {
        DraggingConstraint constraint = getDraggingConstraintWith2LockedCorners(
            lockedCorner1, lockedPos1,
            lockedCorner2, lockedPos2,
            freeCorner
        );
        
        class_243 freePosLimited = constraint.constrain(freePos);
        
        if (freePosLimited == null) {
            return null;
        }
        
        // determine the 4 vertices of the portal
        class_243[][] vertices = new class_243[2][2];
        
        int corner1XSign = lockedCorner1.getXSign();
        int corner1YSign = lockedCorner1.getYSign();
        int corner2XSign = lockedCorner2.getXSign();
        int corner2YSign = lockedCorner2.getYSign();
        int freeCornerXSign = freeCorner.getXSign();
        int freeCornerYSign = freeCorner.getYSign();
        
        vertices[corner1XSign == -1 ? 0 : 1][corner1YSign == -1 ? 0 : 1] = lockedPos1;
        vertices[corner2XSign == -1 ? 0 : 1][corner2YSign == -1 ? 0 : 1] = lockedPos2;
        vertices[freeCornerXSign == -1 ? 0 : 1][freeCornerYSign == -1 ? 0 : 1] = freePosLimited;
        
        for (int cx = 0; cx <= 1; cx++) {
            for (int cy = 0; cy <= 1; cy++) {
                if (vertices[cx][cy] == null) {
                    class_243 sideVertex1 = vertices[1 - cx][cy];
                    class_243 sideVertex2 = vertices[cx][1 - cy];
                    class_243 diagonalVertex = vertices[1 - cx][1 - cy];
                    class_243 v1 = sideVertex1.method_1020(diagonalVertex);
                    class_243 v2 = sideVertex2.method_1020(diagonalVertex);
                    vertices[cx][cy] = diagonalVertex.method_1019(v1).method_1019(v2);
                }
            }
        }
        
        class_243 horizontalAxis = vertices[1][0].method_1020(vertices[0][0]);
        class_243 verticalAxis = vertices[0][1].method_1020(vertices[0][0]);
        class_243 axisW = horizontalAxis.method_1029();
        class_243 axisH = verticalAxis.method_1029();
        class_243 normal = axisW.method_1036(axisH);
        
        if (Math.abs(axisW.method_1026(axisH)) > 0.01) {
            LOGGER.error("The dragged portal vertices are ill-formed");
            return null;
        }
        
        DQuaternion orientation = DQuaternion.matrixToQuaternion(axisW, axisH, normal);
        
        return new UnilateralPortalState.Builder()
            .dimension(originalState.dimension())
            .position(vertices[0][0].method_1019(vertices[1][1]).method_1021(0.5))
            .orientation(orientation)
            .width(horizontalAxis.method_1033())
            .height(verticalAxis.method_1033())
            .build();
    }
}
