package qouteall.imm_ptl.peripheral.wand;

import com.mojang.logging.LogUtils;
import org.apache.commons.lang3.Validate;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import qouteall.imm_ptl.core.IPGlobal;
import qouteall.imm_ptl.core.McHelper;
import qouteall.imm_ptl.core.portal.Portal;
import qouteall.imm_ptl.core.portal.PortalExtension;
import qouteall.imm_ptl.core.portal.PortalManipulation;
import qouteall.imm_ptl.core.portal.PortalState;
import qouteall.imm_ptl.core.portal.animation.UnilateralPortalState;
import qouteall.imm_ptl.core.portal.util.PortalLocalXYNormalized;
import qouteall.imm_ptl.peripheral.CommandStickItem;
import qouteall.q_misc_util.my_util.DQuaternion;
import qouteall.q_misc_util.my_util.Plane;
import qouteall.q_misc_util.my_util.Range;

import java.util.List;
import java.util.UUID;
import java.util.WeakHashMap;
import net.minecraft.class_1297;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1937;
import net.minecraft.class_243;
import net.minecraft.class_2487;
import net.minecraft.class_2561;
import net.minecraft.class_3218;
import net.minecraft.class_3222;
import net.minecraft.class_5321;

public class PortalWandInteraction {
    
    private static final double SIZE_LIMIT = 64;
    
    private static final Logger LOGGER = LogUtils.getLogger();
    
    public static class RemoteCallables {
        public static void finishPortalCreation(
            class_3222 player,
            ProtoPortal protoPortal
        ) {
            if (!checkPermission(player)) return;
            
            handleFinishPortalCreation(player, protoPortal);
        }
        
        public static void requestApplyDrag(
            class_3222 player,
            UUID portalId,
            class_243 cursorPos,
            DraggingInfo draggingInfo
        ) {
            if (!checkPermission(player)) return;
            
            class_1297 entity = ((class_3218) player.method_37908()).method_14190(portalId);
            
            if (!(entity instanceof Portal portal)) {
                LOGGER.error("Cannot find portal {}", portalId);
                return;
            }
            
            if (!draggingInfo.isValid()) {
                player.method_43496(class_2561.method_43470("Invalid dragging info"));
                LOGGER.error("Invalid dragging info {}", draggingInfo);
                return;
            }
            
            handleDraggingRequest(player, portalId, cursorPos, draggingInfo, portal);
        }
        
        public static void undoDrag(class_3222 player) {
            if (!checkPermission(player)) return;
            
            handleUndoDrag(player);
        }
        
        public static void finishDragging(class_3222 player) {
            if (!checkPermission(player)) return;
            
            handleFinishDrag(player);
        }
        
        public static void copyCutPortal(class_3222 player, UUID portalId, boolean isCut) {
            if (!checkPermission(player)) return;
            
            handleCopyCutPortal(player, portalId, isCut);
        }
        
        public static void confirmCopyCut(class_3222 player, class_243 origin, DQuaternion orientation) {
            if (!checkPermission(player)) return;
            
            handleConfirmCopyCut(player, origin, orientation);
        }
        
        public static void clearPortalClipboard(class_3222 player) {
            if (!checkPermission(player)) return;
            
            handleClearPortalClipboard(player);
        }
    }
    
    private static void handleFinishPortalCreation(class_3222 player, ProtoPortal protoPortal) {
        Validate.isTrue(protoPortal.firstSide != null);
        Validate.isTrue(protoPortal.secondSide != null);
        
        class_243 firstSideLeftBottom = protoPortal.firstSide.leftBottom;
        class_243 firstSideRightBottom = protoPortal.firstSide.rightBottom;
        class_243 firstSideLeftUp = protoPortal.firstSide.leftTop;
        class_243 secondSideLeftBottom = protoPortal.secondSide.leftBottom;
        class_243 secondSideRightBottom = protoPortal.secondSide.rightBottom;
        class_243 secondSideLeftUp = protoPortal.secondSide.leftTop;
        
        Validate.notNull(firstSideLeftBottom);
        Validate.notNull(firstSideRightBottom);
        Validate.notNull(firstSideLeftUp);
        Validate.notNull(secondSideLeftBottom);
        Validate.notNull(secondSideRightBottom);
        Validate.notNull(secondSideLeftUp);
        
        class_5321<class_1937> firstSideDimension = protoPortal.firstSide.dimension;
        class_5321<class_1937> secondSideDimension = protoPortal.secondSide.dimension;
        
        class_243 firstSideHorizontalAxis = firstSideRightBottom.method_1020(firstSideLeftBottom);
        class_243 firstSideVerticalAxis = firstSideLeftUp.method_1020(firstSideLeftBottom);
        double firstSideWidth = firstSideHorizontalAxis.method_1033();
        double firstSideHeight = firstSideVerticalAxis.method_1033();
        class_243 firstSideHorizontalUnitAxis = firstSideHorizontalAxis.method_1029();
        class_243 firstSideVerticalUnitAxis = firstSideVerticalAxis.method_1029();
        
        if (Math.abs(firstSideWidth) < 0.001 || Math.abs(firstSideHeight) < 0.001) {
            player.method_43496(class_2561.method_43470("The first side is too small"));
            LOGGER.error("The first side is too small");
            return;
        }
        
        if (firstSideHorizontalUnitAxis.method_1026(firstSideVerticalUnitAxis) > 0.001) {
            player.method_43496(class_2561.method_43470("The horizontal and vertical axis are not perpendicular in first side"));
            LOGGER.error("The horizontal and vertical axis are not perpendicular in first side");
            return;
        }
        
        if (firstSideWidth > SIZE_LIMIT || firstSideHeight > SIZE_LIMIT) {
            player.method_43496(class_2561.method_43470("The first side is too large"));
            LOGGER.error("The first side is too large");
            return;
        }
        
        class_243 secondSideHorizontalAxis = secondSideRightBottom.method_1020(secondSideLeftBottom);
        class_243 secondSideVerticalAxis = secondSideLeftUp.method_1020(secondSideLeftBottom);
        double secondSideWidth = secondSideHorizontalAxis.method_1033();
        double secondSideHeight = secondSideVerticalAxis.method_1033();
        class_243 secondSideHorizontalUnitAxis = secondSideHorizontalAxis.method_1029();
        class_243 secondSideVerticalUnitAxis = secondSideVerticalAxis.method_1029();
        
        if (Math.abs(secondSideWidth) < 0.001 || Math.abs(secondSideHeight) < 0.001) {
            player.method_43496(class_2561.method_43470("The second side is too small"));
            LOGGER.error("The second side is too small");
            return;
        }
        
        if (secondSideHorizontalUnitAxis.method_1026(secondSideVerticalUnitAxis) > 0.001) {
            player.method_43496(class_2561.method_43470("The horizontal and vertical axis are not perpendicular in second side"));
            LOGGER.error("The horizontal and vertical axis are not perpendicular in second side");
            return;
        }
        
        if (secondSideWidth > SIZE_LIMIT || secondSideHeight > SIZE_LIMIT) {
            player.method_43496(class_2561.method_43470("The second side is too large"));
            LOGGER.error("The second side is too large");
            return;
        }
        
        if (Math.abs((firstSideHeight / firstSideWidth) - (secondSideHeight / secondSideWidth)) > 0.001) {
            player.method_43496(class_2561.method_43470("The two sides have different aspect ratio"));
            LOGGER.error("The two sides have different aspect ratio");
            return;
        }
        
        boolean overlaps = false;
        if (firstSideDimension == secondSideDimension) {
            class_243 firstSideNormal = firstSideHorizontalUnitAxis.method_1036(firstSideVerticalUnitAxis);
            class_243 secondSideNormal = secondSideHorizontalUnitAxis.method_1036(secondSideVerticalUnitAxis);
            
            // check orientation overlap
            if (Math.abs(firstSideNormal.method_1026(secondSideNormal)) > 0.99) {
                // check plane overlap
                if (Math.abs(firstSideLeftBottom.method_1020(secondSideLeftBottom).method_1026(firstSideNormal)) < 0.001) {
                    // check portal area overlap
                    
                    class_243 coordCenter = firstSideLeftBottom;
                    class_243 coordX = firstSideHorizontalAxis;
                    class_243 coordY = firstSideVerticalAxis;
                    
                    Range firstSideXRange = Range.createUnordered(
                        firstSideLeftBottom.method_1020(coordCenter).method_1026(coordX),
                        firstSideRightBottom.method_1020(coordCenter).method_1026(coordX)
                    );
                    Range firstSideYRange = Range.createUnordered(
                        firstSideLeftBottom.method_1020(coordCenter).method_1026(coordY),
                        firstSideLeftUp.method_1020(coordCenter).method_1026(coordY)
                    );
                    
                    Range secondSideXRange = Range.createUnordered(
                        secondSideLeftBottom.method_1020(coordCenter).method_1026(coordX),
                        secondSideRightBottom.method_1020(coordCenter).method_1026(coordX)
                    );
                    Range secondSideYRange = Range.createUnordered(
                        secondSideLeftBottom.method_1020(coordCenter).method_1026(coordY),
                        secondSideLeftUp.method_1020(coordCenter).method_1026(coordY)
                    );
                    
                    if (firstSideXRange.intersect(secondSideXRange) != null &&
                        firstSideYRange.intersect(secondSideYRange) != null
                    ) {
                        overlaps = true;
                    }
                }
            }
        }
        
        Portal portal = Portal.entityType.method_5883(McHelper.getServerWorld(firstSideDimension));
        Validate.notNull(portal);
        portal.setOriginPos(
            firstSideLeftBottom
                .method_1019(firstSideHorizontalAxis.method_1021(0.5))
                .method_1019(firstSideVerticalAxis.method_1021(0.5))
        );
        portal.width = firstSideWidth;
        portal.height = firstSideHeight;
        portal.axisW = firstSideHorizontalUnitAxis;
        portal.axisH = firstSideVerticalUnitAxis;
        
        portal.dimensionTo = secondSideDimension;
        portal.setDestination(
            secondSideLeftBottom
                .method_1019(secondSideHorizontalAxis.method_1021(0.5))
                .method_1019(secondSideVerticalAxis.method_1021(0.5))
        );
        
        portal.scaling = secondSideWidth / firstSideWidth;
        
        DQuaternion secondSideOrientation = DQuaternion.matrixToQuaternion(
            secondSideHorizontalUnitAxis,
            secondSideVerticalUnitAxis,
            secondSideHorizontalUnitAxis.method_1036(secondSideVerticalUnitAxis)
        );
        portal.setOtherSideOrientation(secondSideOrientation);
        
        Portal flippedPortal = PortalManipulation.createFlippedPortal(portal, Portal.entityType);
        Portal reversePortal = PortalManipulation.createReversePortal(portal, Portal.entityType);
        Portal parallelPortal = PortalManipulation.createFlippedPortal(reversePortal, Portal.entityType);
        
        McHelper.spawnServerEntity(portal);
        
        if (overlaps) {
            player.method_43496(class_2561.method_43471("imm_ptl.wand.overlap"));
        }
        else {
            McHelper.spawnServerEntity(flippedPortal);
            McHelper.spawnServerEntity(reversePortal);
            McHelper.spawnServerEntity(parallelPortal);
        }
        
        player.method_43496(class_2561.method_43471("imm_ptl.wand.finished"));
        
        giveCommandStick(player, "/portal eradicate_portal_cluster");
    }
    
    private static class DraggingSession {
        private final class_5321<class_1937> dimension;
        private final UUID portalId;
        private final PortalState originalState;
        private final DraggingInfo lastDraggingInfo;
        
        public DraggingSession(
            class_5321<class_1937> dimension, UUID portalId,
            PortalState originalState, DraggingInfo lastDraggingInfo
        ) {
            this.dimension = dimension;
            this.portalId = portalId;
            this.originalState = originalState;
            this.lastDraggingInfo = lastDraggingInfo;
        }
        
        @Nullable
        public Portal getPortal() {
            class_1297 entity = McHelper.getServerWorld(dimension).method_14190(portalId);
            
            if (entity instanceof Portal) {
                return (Portal) entity;
            }
            else {
                return null;
            }
        }
    }
    
    private static final WeakHashMap<class_3222, DraggingSession> draggingSessionMap = new WeakHashMap<>();
    
    public static void init() {
        IPGlobal.postServerTickSignal.connect(() -> {
            draggingSessionMap.entrySet().removeIf(
                e -> {
                    class_3222 player = e.getKey();
                    if (player.method_31481()) {
                        return true;
                    }
                    
                    if (player.method_6047().method_7909() != PortalWandItem.instance) {
                        return true;
                    }
                    
                    return false;
                }
            );
            
            copyingSessionMap.entrySet().removeIf(
                e -> {
                    class_3222 player = e.getKey();
                    return player.method_31481();
                }
            );
        });
        
        IPGlobal.serverCleanupSignal.connect(draggingSessionMap::clear);
        IPGlobal.serverCleanupSignal.connect(copyingSessionMap::clear);
    }
    
    public static final class DraggingInfo {
        public final @Nullable PortalLocalXYNormalized lockedAnchor;
        public final PortalLocalXYNormalized draggingAnchor;
        public @Nullable class_243 previousRotationAxis;
        public final boolean lockWidth;
        public final boolean lockHeight;
        
        public DraggingInfo(
            @Nullable PortalLocalXYNormalized lockedAnchor,
            PortalLocalXYNormalized draggingAnchor,
            @Nullable class_243 previousRotationAxis,
            boolean lockWidth,
            boolean lockHeight
        ) {
            this.lockedAnchor = lockedAnchor;
            this.draggingAnchor = draggingAnchor;
            this.previousRotationAxis = previousRotationAxis;
            this.lockWidth = lockWidth;
            this.lockHeight = lockHeight;
        }
        
        public boolean shouldLockScale() {
            return lockWidth || lockHeight;
        }
        
        public boolean isValid() {
            if (lockedAnchor != null) {
                if (!lockedAnchor.isValid()) {
                    return false;
                }
            }
            if (!draggingAnchor.isValid()) {
                return false;
            }
            return true;
        }
    }
    
    @Nullable
    public static UnilateralPortalState applyDrag(
        UnilateralPortalState originalState, class_243 cursorPos, DraggingInfo info,
        boolean updateInternalState
    ) {
        if (info.lockedAnchor == null) {
            class_243 offset = info.draggingAnchor.getOffset(originalState);
            class_243 newPos = cursorPos.method_1020(offset);
            
            return new UnilateralPortalState.Builder()
                .from(originalState)
                .position(newPos)
                .build();
        }
        
        OneLockDraggingResult r = performDragWithOneLockedAnchor(
            originalState, info.lockedAnchor, info.draggingAnchor, cursorPos, info.previousRotationAxis,
            info.lockWidth, info.lockHeight
        );
        
        if (r == null) {
            return null;
        }
        
        if (updateInternalState) {
            info.previousRotationAxis = r.rotationAxis();
        }
        return r.newState();
    }
    
    private static void handleFinishDrag(class_3222 player) {
        DraggingSession session = draggingSessionMap.remove(player);
        
        if (session == null) {
            return;
        }
        
        Portal portal = session.getPortal();
        
        if (portal != null) {
            portal.reloadAndSyncToClientNextTick();
        }
    }
    
    private static void handleUndoDrag(class_3222 player) {
        DraggingSession session = draggingSessionMap.get(player);
        
        if (session == null) {
//            player.sendSystemMessage(Component.literal("Cannot undo"));
            return;
        }
        
        Portal portal = session.getPortal();
        
        if (portal == null) {
            LOGGER.error("Cannot find portal {}", session.portalId);
            return;
        }
        
        portal.setPortalState(session.originalState);
        portal.reloadAndSyncToClientNextTick();
        portal.rectifyClusterPortals(true);
        
        draggingSessionMap.remove(player);
    }
    
    private static void handleDraggingRequest(
        class_3222 player, UUID portalId, class_243 cursorPos, DraggingInfo draggingInfo, Portal portal
    ) {
        DraggingSession session = draggingSessionMap.get(player);
        
        if (session != null && session.portalId.equals(portalId)) {
            // reuse session
        }
        else {
            session = new DraggingSession(
                player.method_37908().method_27983(),
                portalId,
                portal.getPortalState(),
                draggingInfo
            );
            draggingSessionMap.put(player, session);
        }
        
        UnilateralPortalState newThisSideState = applyDrag(
            session.originalState.getThisSideState(), cursorPos, draggingInfo,
            true
        );
        if (validateDraggedPortalState(session.originalState, newThisSideState, player)) {
            portal.setThisSideState(newThisSideState, draggingInfo.shouldLockScale());
            portal.reloadAndSyncToClientNextTick();
            portal.rectifyClusterPortals(true);
        }
        else {
            player.method_43496(class_2561.method_43470("Invalid dragging"));
        }
    }
    
    private static boolean checkPermission(class_3222 player) {
        if (!canPlayerUsePortalWand(player)) {
            player.method_43496(class_2561.method_43470("You cannot use portal wand"));
            LOGGER.error("Player cannot use portal wand {}", player);
            return false;
        }
        return true;
    }
    
    public static boolean validateDraggedPortalState(
        PortalState originalState, UnilateralPortalState newThisSideState,
        class_1657 player
    ) {
        if (newThisSideState == null) {
            return false;
        }
        if (newThisSideState.width() > 64.1) {
            return false;
        }
        if (newThisSideState.height() > 64.1) {
            return false;
        }
        if (newThisSideState.width() < 0.05) {
            return false;
        }
        if (newThisSideState.height() < 0.05) {
            return false;
        }
        
        if (originalState.fromWorld != newThisSideState.dimension()) {
            return false;
        }
        
        if (newThisSideState.position().method_1022(player.method_19538()) > 64) {
            return false;
        }
        
        return true;
    }
    
    private static boolean canPlayerUsePortalWand(class_3222 player) {
        return player.method_5687(2) || (IPGlobal.easeCreativePermission && player.method_7337());
    }
    
    private static void giveCommandStick(class_3222 player, String command) {
        CommandStickItem.Data data = CommandStickItem.BUILT_IN_COMMAND_STICK_TYPES.get(command);
        
        if (data == null) {
            data = new CommandStickItem.Data(command, command, List.of());
        }
        
        class_1799 stack = new class_1799(CommandStickItem.instance);
        stack.method_7980(data.toTag());
        
        if (!player.method_31548().method_7379(stack)) {
            player.method_31548().method_7394(stack);
        }
    }
    
    public static boolean isDragging(class_3222 player) {
        return draggingSessionMap.containsKey(player);
    }
    
    @Nullable
    public static PortalWandInteraction.OneLockDraggingResult performDragWithOneLockedAnchor(
        UnilateralPortalState originalState,
        PortalLocalXYNormalized lockedLocalPos,
        PortalLocalXYNormalized draggingLocalPos, class_243 draggedPos,
        @Nullable class_243 previousRotationAxis,
        boolean lockWidth, boolean lockHeight
    ) {
        class_243 draggedPosOriginalPos = draggingLocalPos.getPos(originalState);
        class_243 lockedPos = lockedLocalPos.getPos(originalState);
        class_243 originalOffset = draggedPosOriginalPos.method_1020(lockedPos);
        class_243 newOffset = draggedPos.method_1020(lockedPos);
        
        double newOffsetLen = newOffset.method_1033();
        double originalOffsetLen = originalOffset.method_1033();
        
        if (newOffsetLen < 0.00001 || originalOffsetLen < 0.00001) {
            return null;
        }
        
        class_243 originalOffsetN = originalOffset.method_1029();
        class_243 newOffsetN = newOffset.method_1029();
        
        double dot = originalOffsetN.method_1026(newOffsetN);
        
        DQuaternion rotation;
        if (Math.abs(dot) < 0.99999) {
            rotation = DQuaternion.getRotationBetween(originalOffset, newOffset)
                .fixFloatingPointErrorAccumulation();
        }
        else {
            // the originalOffset and newOffset are colinear
            
            if (dot > 0) {
                // the two offsets are roughly equal. no dragging
                rotation = DQuaternion.identity;
            }
            else {
                // the two offsets are opposite.
                // we cannot determine the rotation axis. the possible axises are on a plane
                
                Plane planeOfPossibleAxis = new Plane(class_243.field_1353, originalOffsetN);
                
                // to improve user-friendliness, use the axis from the previous rotation
                if (previousRotationAxis != null) {
                    // project the previous axis onto the plane of possible axis
                    class_243 projected = planeOfPossibleAxis.getProjection(previousRotationAxis);
                    if (projected.method_1027() < 0.00001) {
                        // the previous axis is perpendicular to the plane
                        // cannot determine axis
                        return null;
                    }
                    class_243 axis = projected.method_1029();
                    rotation = DQuaternion.rotationByDegrees(axis, 180)
                        .fixFloatingPointErrorAccumulation();
                }
                else {
                    rotation = DQuaternion.identity;
                }
            }
        }
        
        DQuaternion newOrientation = rotation.hamiltonProduct(originalState.orientation())
            .fixFloatingPointErrorAccumulation();
        
        PortalLocalXYNormalized deltaLocalXY = draggingLocalPos.subtract(lockedLocalPos);
        
        double newWidth;
        double newHeight;
        if (lockWidth && lockHeight) {
            newWidth = originalState.width();
            newHeight = originalState.height();
        }
        else {
            class_243 newNormal = rotation.rotate(originalState.getNormal());
            if (lockWidth) {
                assert !lockHeight;
                newWidth = originalState.width();
                if (Math.abs(deltaLocalXY.ny()) < 0.001) {
                    newHeight = originalState.height();
                }
                else {
                    double subWidth = Math.abs(deltaLocalXY.nx()) * newWidth;
                    double diff = newOffsetLen * newOffsetLen - subWidth * subWidth;
                    if (diff < 0.000001) {
                        return null;
                    }
                    double subHeight = Math.sqrt(diff);
                    newHeight = subHeight / Math.abs(deltaLocalXY.ny());
                    if (Math.abs(subWidth) > 0.001) {
                        // if dragging on non-axis-aligned direction and aspect ratio changes,
                        // the simple rotation will not match the dragging cursor.
                        // recalculate the orientation
                        newOrientation = getOrientationByNormalDiagonalWidthHeight(
                            newNormal, newOffset, subWidth, subHeight,
                            Math.signum(deltaLocalXY.nx()), Math.signum(deltaLocalXY.ny())
                        );
                    }
                }
            }
            else if (lockHeight) {
                assert !lockWidth;
                newHeight = originalState.height();
                if (Math.abs(deltaLocalXY.nx()) < 0.001) {
                    newWidth = originalState.width();
                }
                else {
                    double subHeight = Math.abs(deltaLocalXY.ny()) * newHeight;
                    double diff = newOffsetLen * newOffsetLen - subHeight * subHeight;
                    if (diff < 0.000001) {
                        return null;
                    }
                    double subWidth = Math.sqrt(diff);
                    newWidth = subWidth / Math.abs(deltaLocalXY.nx());
                    if (Math.abs(subHeight) > 0.001) {
                        // if dragging on non-axis-aligned direction and aspect ratio changes,
                        // the simple rotation will not match the dragging cursor.
                        // recalculate the orientation
                        newOrientation = getOrientationByNormalDiagonalWidthHeight(
                            newNormal, newOffset, subWidth, subHeight,
                            Math.signum(deltaLocalXY.nx()), Math.signum(deltaLocalXY.ny())
                        );
                    }
                }
            }
            else {
                double scaling = newOffsetLen / originalOffsetLen;
                newWidth = originalState.width() * scaling;
                newHeight = originalState.height() * scaling;
            }
        }
        
        // get the new unilateral portal state by scaling and rotation
        class_243 newLockedPosOffset = newOrientation.rotate(new class_243(
            (lockedLocalPos.nx() - 0.5) * newWidth,
            (lockedLocalPos.ny() - 0.5) * newHeight,
            0
        ));
        class_243 newOrigin = lockedPos.method_1020(newLockedPosOffset);
        
        UnilateralPortalState newPortalState = new UnilateralPortalState(
            originalState.dimension(), newOrigin, newOrientation, newWidth, newHeight
        );
        return new OneLockDraggingResult(
            newPortalState, rotation.getRotatingAxis()
        );
    }
    
    /**
     * The sub-portal defined by the locked pos and the cursor pos
     * has the same orientation as the outer portal.
     * Get the sub-portal's orientation by normal, diagonal, width and height.
     */
    private static DQuaternion getOrientationByNormalDiagonalWidthHeight(
        class_243 normal, class_243 diagonal,
        double width, double height,
        double sigX, double sigY
    ) {
        class_243 newOffsetN = diagonal.method_1029();
        double newOffsetLen = diagonal.method_1033();
        
        // the side vec is on the left side of the diagonal (right-hand rule)
        class_243 sideVecN = normal.method_1036(newOffsetN).method_1029();
        
        // the triangle area can be represented by two ways:
        // diagonalLen * sideVecLen / 2, or width * height / 2
        double sideVecLen = (width * height) / newOffsetLen;
        
        // the tangent can be represented by two ways:
        // width / height, or sideVecLen / wFront
        double wFront = sideVecLen * width / height;
        double hFront = sideVecLen * height / width;
        
        // if sigX and sigY are both 1,
        // then the axisW is on the right side of the diagonal (right-hand rule),
        // and axisH is on the left side of the diagonal.
        // if either sigX or sigY flips, this flips, so multiply the signum.
        class_243 sideVecW = sideVecN.method_1021(-sideVecLen * sigX * sigY);
        class_243 sideVecH = sideVecW.method_1021(-1);
        
        class_243 newAxisW = newOffsetN.method_1021(wFront)
            .method_1019(sideVecW).method_1029()
            .method_1021(sigX); // make it the same direction of the outer portal
        
        class_243 newAxisH = newOffsetN.method_1021(hFront)
            .method_1019(sideVecH).method_1029()
            .method_1021(sigY);
        
        DQuaternion newOrientation = DQuaternion.fromFacingVecs(newAxisW, newAxisH)
            .fixFloatingPointErrorAccumulation();
        return newOrientation;
    }
    
    public static record OneLockDraggingResult(
        UnilateralPortalState newState,
        class_243 rotationAxis
    ) {
    }
    
    private record CopyingSession(
        PortalState portalState, class_2487 portalData, boolean isCut,
        boolean hasFlipped, boolean hasReverse, boolean hasParallel
        // TODO store animation view
    ) {
    }
    
    private static final WeakHashMap<class_3222, CopyingSession> copyingSessionMap = new WeakHashMap<>();
    
    public static void handleCopyCutPortal(class_3222 player, UUID portalId, boolean isCut) {
        Portal portal = WandUtil.getPortalByUUID(player.method_37908(), portalId);
        
        if (portal == null) {
            player.method_43496(class_2561.method_43470("Cannot find portal " + portalId));
            return;
        }
        
        PortalState portalState = portal.getPortalState();
        class_2487 portalData = portal.writePortalDataToNbt();
        
        Portal flipped = null;
        Portal reverse = null;
        Portal parallel = null;
        
        PortalExtension ext = PortalExtension.get(portal);
        if (ext.flippedPortal != null) {
            flipped = ext.flippedPortal;
        }
        if (ext.reversePortal != null) {
            reverse = ext.reversePortal;
        }
        if (ext.parallelPortal != null) {
            parallel = ext.parallelPortal;
        }
        
        CopyingSession copyingSession = new CopyingSession(
            portalState, portalData, isCut,
            flipped != null, reverse != null, parallel != null
        );
        copyingSessionMap.put(player, copyingSession);
        
        if (isCut) {
            portal.method_5650(class_1297.class_5529.field_26998);
            if (flipped != null) {
                flipped.method_5650(class_1297.class_5529.field_26998);
            }
            if (reverse != null) {
                reverse.method_5650(class_1297.class_5529.field_26998);
            }
            if (parallel != null) {
                parallel.method_5650(class_1297.class_5529.field_26998);
            }
        }
    }
    
    private static void handleConfirmCopyCut(class_3222 player, class_243 origin, DQuaternion rawOrientation) {
        CopyingSession copyingSession = copyingSessionMap.remove(player);
        
        if (copyingSession == null) {
            player.method_43496(class_2561.method_43470("Missing copying session"));
            return;
        }
        
        DQuaternion orientation = rawOrientation.fixFloatingPointErrorAccumulation();
        
        if (player.method_19538().method_1025(origin) > 64 * 64) {
            player.method_43496(class_2561.method_43470("Too far away from the portal"));
            return;
        }
        
        Portal portal = Portal.entityType.method_5883(player.method_37908());
        assert portal != null;
        
        portal.readPortalDataFromNbt(copyingSession.portalData);
        
        PortalState originalPortalState = copyingSession.portalState;
        
        UnilateralPortalState originalThisSide = originalPortalState.getThisSideState();
        UnilateralPortalState originalOtherSide = originalPortalState.getOtherSideState();
        
        UnilateralPortalState newThisSide = new UnilateralPortalState(
            player.method_37908().method_27983(),
            origin,
            orientation,
            originalThisSide.width(), originalThisSide.height()
        );
        
        portal.setPortalState(UnilateralPortalState.combine(
            newThisSide, originalOtherSide
        ));
        portal.resetAnimationReferenceState(true, false);
        
        if (copyingSession.isCut()) {
            McHelper.spawnServerEntity(portal);
            
            if (copyingSession.hasFlipped) {
                Portal flippedPortal = PortalManipulation.createFlippedPortal(portal, Portal.entityType);
                flippedPortal.resetAnimationReferenceState(true, false);
                McHelper.spawnServerEntity(flippedPortal);
            }
            
            if (copyingSession.hasReverse) {
                Portal reversePortal = PortalManipulation.createReversePortal(portal, Portal.entityType);
                reversePortal.resetAnimationReferenceState(false, true);
                McHelper.spawnServerEntity(reversePortal);
                
                if (copyingSession.hasParallel) {
                    Portal parallelPortal = PortalManipulation.createFlippedPortal(reversePortal, Portal.entityType);
                    parallelPortal.resetAnimationReferenceState(false, true);
                    McHelper.spawnServerEntity(parallelPortal);
                }
            }
        }
        else {
            PortalExtension.get(portal).bindCluster = false;
            McHelper.spawnServerEntity(portal);
            
            if (copyingSession.hasFlipped || copyingSession.hasReverse || copyingSession.hasParallel) {
                player.method_43496(class_2561.method_43471("imm_ptl.wand.copy.not_copying_cluster"));
                giveCommandStick(player, "/portal complete_bi_way_bi_faced_portal");
            }
        }
    }
    
    private static void handleClearPortalClipboard(class_3222 player) {
        copyingSessionMap.remove(player);
    }
    
    
}
