package uk.co.cablepost.bb_boat_hud.webui;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import me.shedaniel.autoconfig.AutoConfig;
import net.minecraft.class_310;
import uk.co.cablepost.bb_boat_hud.client.AnchorType;
import uk.co.cablepost.bb_boat_hud.client.BbBoatHudClient;
import uk.co.cablepost.bb_boat_hud.client.HudModulePlacement;
import uk.co.cablepost.bb_boat_hud.config.ModConfig;

import java.util.*;

public class ConfigWebServer {
    private static Thread serverThread;
    private static volatile boolean running = false;
    private static final int PORT = 8765;
    private static final Gson gson = new GsonBuilder().setPrettyPrinting().create();

    public static void start() {
        if (running) {
            return;
        }

        running = true;
        serverThread = new Thread(() -> {
            try (var serverSocket = new java.net.ServerSocket(PORT)) {
                System.out.println("BB Boat HUD Config Server started on http://localhost:" + PORT);
                
                while (running) {
                    try {
                        var socket = serverSocket.accept();
                        new Thread(() -> handleClient(socket)).start();
                    } catch (Exception e) {
                        if (running) {
                            System.err.println("Error accepting connection: " + e.getMessage());
                        }
                    }
                }
            } catch (Exception e) {
                System.err.println("Failed to start config web server: " + e.getMessage());
            }
        });
        serverThread.setDaemon(true);
        serverThread.start();
    }

    private static void handleClient(java.net.Socket socket) {
        try (socket;
             var in = new java.io.BufferedReader(new java.io.InputStreamReader(socket.getInputStream()));
             var out = new java.io.PrintWriter(socket.getOutputStream(), true)) {
            
            String requestLine = in.readLine();
            if (requestLine == null) return;
            
            String[] parts = requestLine.split(" ");
            if (parts.length < 2) return;
            
            String method = parts[0];
            String path = parts[1];
            
            Map<String, String> headers = new HashMap<>();
            String line;
            while ((line = in.readLine()) != null && !line.isEmpty()) {
                int colonIndex = line.indexOf(':');
                if (colonIndex > 0) {
                    headers.put(line.substring(0, colonIndex).trim().toLowerCase(), 
                              line.substring(colonIndex + 1).trim());
                }
            }
            
            String body = null;
            if (headers.containsKey("content-length")) {
                int contentLength = Integer.parseInt(headers.get("content-length"));
                char[] bodyChars = new char[contentLength];
                in.read(bodyChars, 0, contentLength);
                body = new String(bodyChars);
            }
            
            handleRequest(method, path, body, out);
            
        } catch (Exception e) {
            System.err.println("Error handling client: " + e.getMessage());
        }
    }
    
    private static void handleRequest(String method, String path, String body, java.io.PrintWriter out) {
        try {
            if ("OPTIONS".equals(method)) {
                sendResponse(out, 200, "", "text/plain");
                return;
            }
            
            if ("/".equals(path) && "GET".equals(method)) {
                sendResponse(out, 200, getIndexHtml(), "text/html");
            } else if ("/api/config".equals(path) && "GET".equals(method)) {
                ModConfig config = AutoConfig.getConfigHolder(ModConfig.class).getConfig();
                String json = gson.toJson(config);
                sendResponse(out, 200, json, "application/json");
            } else if ("/api/modules".equals(path) && "GET".equals(method)) {
                Map<String, List<String>> modules = new HashMap<>();
                for (var entry : BbBoatHudClient.HUD_MODULES.entrySet()) {
                    String name = entry.getKey().toString();
                    String[] splitByColon = name.split(":");
                    List<String> pathParts = new ArrayList<>();
                    pathParts.add(splitByColon[0]);
                    if (splitByColon.length > 1) {
                        pathParts.addAll(Arrays.stream(splitByColon[1].split("/")).skip(1).toList());
                    }
                    modules.put(name, pathParts);
                }
                String json = gson.toJson(modules);
                sendResponse(out, 200, json, "application/json");
            } else if ("/api/modules/placements".equals(path) && "POST".equals(method)) {
                HudModulePlacement[] placements = gson.fromJson(body, HudModulePlacement[].class);
                
                // Execute on client thread to avoid ConcurrentModificationException
                class_310.method_1551().execute(() -> {
                    ModConfig config = AutoConfig.getConfigHolder(ModConfig.class).getConfig();
                    config.modulePlacements = new ArrayList<>(Arrays.asList(placements));
                    AutoConfig.getConfigHolder(ModConfig.class).save();
                });
                
                sendResponse(out, 200, "{\"success\": true}", "application/json");
            } else if ("/api/modules/add".equals(path) && "POST".equals(method)) {
                @SuppressWarnings("unchecked")
                Map<String, String> data = gson.fromJson(body, Map.class);
                String identifier = data.get("identifier");
                
                // Execute on client thread to avoid ConcurrentModificationException
                class_310.method_1551().execute(() -> {
                    HudModulePlacement placement = new HudModulePlacement();
                    placement.identifier = identifier;
                    placement.anchorType = AnchorType.MIDDLE_CENTER;
                    placement.xOffset = 0;
                    placement.yOffset = 0;
                    placement.scale = 1.0f;
                    placement.angle = 0.0f;
                    
                    ModConfig config = AutoConfig.getConfigHolder(ModConfig.class).getConfig();
                    config.modulePlacements.add(placement);
                    AutoConfig.getConfigHolder(ModConfig.class).save();
                });
                
                sendResponse(out, 200, "{\"success\": true}", "application/json");
            } else if ("/api/vanilla".equals(path) && "POST".equals(method)) {
                ModConfig newConfig = gson.fromJson(body, ModConfig.class);
                
                // Execute on client thread to avoid ConcurrentModificationException
                class_310.method_1551().execute(() -> {
                    ModConfig config = AutoConfig.getConfigHolder(ModConfig.class).getConfig();
                    
                    config.hotbar = newConfig.hotbar;
                    config.experienceBar = newConfig.experienceBar;
                    config.experienceLevel = newConfig.experienceLevel;
                    config.heldItemTooltip = newConfig.heldItemTooltip;
                    config.armorBar = newConfig.armorBar;
                    config.healthBar = newConfig.healthBar;
                    config.foodBar = newConfig.foodBar;
                    config.oxygenBar = newConfig.oxygenBar;
                    config.bossBar = newConfig.bossBar;
                    config.overlayMessage = newConfig.overlayMessage;
                    
                    AutoConfig.getConfigHolder(ModConfig.class).save();
                });
                
                sendResponse(out, 200, "{\"success\": true}", "application/json");
            } else {
                sendResponse(out, 404, "Not Found", "text/plain");
            }
        } catch (Exception e) {
            System.err.println("Error processing request: " + e.getMessage());
            e.printStackTrace();
            sendResponse(out, 500, "Internal Server Error", "text/plain");
        }
    }
    
    private static void sendResponse(java.io.PrintWriter out, int statusCode, String body, String contentType) {
        out.println("HTTP/1.1 " + statusCode + " " + getStatusMessage(statusCode));
        out.println("Content-Type: " + contentType + "; charset=UTF-8");
        out.println("Content-Length: " + body.getBytes(java.nio.charset.StandardCharsets.UTF_8).length);
        out.println("Access-Control-Allow-Origin: *");
        out.println("Access-Control-Allow-Methods: GET, POST, OPTIONS");
        out.println("Access-Control-Allow-Headers: Content-Type");
        out.println("Connection: close");
        out.println();
        out.print(body);
        out.flush();
    }
    
    private static String getStatusMessage(int statusCode) {
        return switch (statusCode) {
            case 200 -> "OK";
            case 404 -> "Not Found";
            case 405 -> "Method Not Allowed";
            case 500 -> "Internal Server Error";
            default -> "Unknown";
        };
    }

    public static void stop() {
        running = false;
        if (serverThread != null) {
            serverThread.interrupt();
            serverThread = null;
        }
        System.out.println("BB Boat HUD Config Server stopped");
    }

    public static boolean isRunning() {
        return running;
    }

    public static int getPort() {
        return PORT;
    }

    private static String getIndexHtml() {
        return WebUI.getHTML();
    }
}
