package uk.co.cablepost.bb_boat_hud.webui;

public class WebUI {
    public static String getHTML() {
        return """
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>BBBoatHUD Configuration</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Courier New', monospace;
            background: #1a1a1a;
            min-height: 100vh;
            padding: 20px;
            color: #fff;
            image-rendering: pixelated;
            image-rendering: -moz-crisp-edges;
            image-rendering: crisp-edges;
        }
        
        .container {
            max-width: 1200px;
            margin: 0 auto;
            background: #2d2d2d;
            border: 4px solid #000;
            box-shadow: 0 0 0 2px #555, 0 8px 24px rgba(0, 0, 0, 0.8);
        }
        
        header {
            background: #1e1e1e;
            color: #fff;
            padding: 20px;
            text-align: center;
            border-bottom: 4px solid #000;
            box-shadow: inset 0 -2px 0 #555;
        }
        
        header h1 {
            font-size: 2em;
            margin-bottom: 5px;
            text-shadow: 3px 3px 0 #000;
            letter-spacing: 2px;
        }
        
        header p {
            font-size: 0.9em;
            color: #aaa;
        }
        
        .tabs {
            display: flex;
            background: #1e1e1e;
            border-bottom: 4px solid #000;
        }
        
        .tab {
            flex: 1;
            padding: 12px 15px;
            background: #3a3a3a;
            border: 2px solid #000;
            border-bottom: none;
            cursor: pointer;
            font-size: 14px;
            font-weight: bold;
            transition: all 0.1s;
            color: #aaa;
            font-family: 'Courier New', monospace;
            text-transform: uppercase;
            margin-right: 2px;
        }
        
        .tab:hover {
            background: #4a4a4a;
            color: #fff;
        }
        
        .tab.active {
            background: #2d2d2d;
            color: #5f5;
            border-bottom: 2px solid #2d2d2d;
        }
        
        .tab-content {
            display: none;
            padding: 20px;
            max-height: 70vh;
            overflow-y: auto;
        }
        
        .tab-content.active {
            display: block;
        }
        
        .module-item {
            background: #1e1e1e;
            border: 3px solid #000;
            box-shadow: inset 0 0 0 1px #555;
            padding: 10px;
            margin-bottom: 8px;
            transition: all 0.1s;
        }
        
        .module-item:hover {
            border-color: #5f5;
            box-shadow: inset 0 0 0 1px #5f5, 0 0 8px rgba(85, 255, 85, 0.3);
        }
        
        .module-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 8px;
            background: #2d2d2d;
            border: 2px solid #000;
            cursor: pointer;
            user-select: none;
        }
        
        .module-header:hover {
            background: #3a3a3a;
        }
        
        .module-title {
            font-size: 1em;
            font-weight: bold;
            color: #5f5;
            text-shadow: 2px 2px 0 #000;
        }
        
        .module-id {
            font-size: 0.75em;
            color: #888;
            font-family: monospace;
        }
        
        .module-details {
            display: none;
            padding: 15px;
            background: #262626;
            border-top: 2px solid #000;
        }
        
        .module-details.expanded {
            display: block;
        }
        
        .expand-indicator {
            color: #5f5;
            font-weight: bold;
            font-size: 1.2em;
        }
        
        .form-group {
            margin-bottom: 12px;
        }
        
        label {
            display: block;
            margin-bottom: 4px;
            font-weight: bold;
            color: #aaa;
            font-size: 0.85em;
            text-transform: uppercase;
        }
        
        input[type="number"],
        input[type="range"],
        select {
            width: 100%;
            padding: 8px;
            background: #1e1e1e;
            border: 2px solid #000;
            box-shadow: inset 0 0 0 1px #555;
            font-size: 13px;
            font-family: 'Courier New', monospace;
            color: #fff;
        }
        
        input[type="number"]:focus,
        select:focus {
            outline: none;
            border-color: #5f5;
            box-shadow: inset 0 0 0 1px #5f5, 0 0 8px rgba(85, 255, 85, 0.3);
        }
        
        input[type="range"] {
            padding: 5px;
            background: #1e1e1e;
        }
        
        .slider-input-group {
            display: grid;
            grid-template-columns: 1fr 80px;
            gap: 8px;
            align-items: center;
        }
        
        .slider-input-group input[type="range"] {
            width: 100%;
        }
        
        .slider-input-group input[type="number"] {
            width: 100%;
            text-align: center;
        }
        
        .checkbox-group {
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        input[type="checkbox"] {
            width: 18px;
            height: 18px;
            cursor: pointer;
        }
        
        button {
            padding: 8px 16px;
            background: #3a3a3a;
            color: #fff;
            border: 2px solid #000;
            box-shadow: inset 0 0 0 1px #555;
            cursor: pointer;
            font-size: 13px;
            font-weight: bold;
            font-family: 'Courier New', monospace;
            text-transform: uppercase;
            transition: all 0.1s;
        }
        
        button:hover {
            background: #4a4a4a;
            box-shadow: inset 0 0 0 1px #5f5, 0 0 8px rgba(85, 255, 85, 0.3);
            border-color: #5f5;
        }
        
        button:active {
            background: #2a2a2a;
        }
        
        button.danger {
            background: #5a1a1a;
            border-color: #f55;
            box-shadow: inset 0 0 0 1px #f55;
        }
        
        button.danger:hover {
            background: #7a2a2a;
            box-shadow: inset 0 0 0 1px #f55, 0 0 8px rgba(255, 85, 85, 0.3);
        }
        
        .anchor-grid {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 6px;
            margin-top: 8px;
        }
        
        .anchor-btn {
            padding: 8px;
            background: #1e1e1e;
            border: 2px solid #000;
            box-shadow: inset 0 0 0 1px #555;
            cursor: pointer;
            transition: all 0.1s;
            font-size: 11px;
            font-family: 'Courier New', monospace;
            color: #aaa;
        }
        
        .anchor-btn:hover {
            border-color: #5f5;
            background: #2a2a2a;
            color: #fff;
        }
        
        .anchor-btn.active {
            background: #2a4a2a;
            color: #5f5;
            border-color: #5f5;
            box-shadow: inset 0 0 0 1px #5f5, 0 0 8px rgba(85, 255, 85, 0.3);
        }
        
        .module-library {
            max-height: 600px;
            overflow-y: auto;
        }
        
        .folder {
            margin-left: 15px;
        }
        
        .folder-header {
            font-weight: bold;
            margin: 8px 0 4px 0;
            color: #ff5;
            text-shadow: 2px 2px 0 #000;
            text-transform: uppercase;
            font-size: 0.9em;
            cursor: pointer;
            user-select: none;
            padding: 6px 8px;
            background: #2a2a2a;
            border: 2px solid #000;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .folder-header:hover {
            background: #3a3a3a;
        }
        
        .folder-content {
            display: none;
            margin-left: 10px;
        }
        
        .folder-content.expanded {
            display: block;
        }
        
        .folder-indicator {
            color: #ff5;
            font-weight: bold;
        }
        
        .module-add-btn {
            display: block;
            width: 100%;
            margin: 3px 0;
            padding: 8px;
            background: #1e1e1e;
            border: 2px solid #000;
            box-shadow: inset 0 0 0 1px #555;
            text-align: left;
            cursor: pointer;
            transition: all 0.1s;
            font-family: 'Courier New', monospace;
            color: #aaa;
            font-size: 0.85em;
        }
        
        .module-add-btn:hover {
            background: #2a4a2a;
            color: #5f5;
            border-color: #5f5;
            box-shadow: inset 0 0 0 1px #5f5, 0 0 8px rgba(85, 255, 85, 0.3);
        }
        
        .save-indicator {
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 12px 20px;
            background: #2a4a2a;
            color: #5f5;
            border: 2px solid #000;
            box-shadow: inset 0 0 0 1px #5f5, 0 0 16px rgba(85, 255, 85, 0.5);
            opacity: 0;
            transition: opacity 0.2s;
            z-index: 1000;
            font-family: 'Courier New', monospace;
            font-weight: bold;
            text-shadow: 2px 2px 0 #000;
        }
        
        .save-indicator.show {
            opacity: 1;
        }
        
        .vanilla-config {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 12px;
        }
        
        .vanilla-item {
            background: #1e1e1e;
            border: 3px solid #000;
            box-shadow: inset 0 0 0 1px #555;
            padding: 12px;
        }
        
        .vanilla-item h3 {
            margin-bottom: 10px;
            color: #ff5;
            text-shadow: 2px 2px 0 #000;
            font-size: 1em;
            text-transform: uppercase;
        }
        
        .empty-state {
            text-align: center;
            padding: 40px 20px;
            color: #666;
        }
        
        .empty-state-icon {
            font-size: 3em;
            margin-bottom: 15px;
            color: #444;
        }
        
        .empty-state p {
            font-size: 1em;
            color: #888;
        }
        
        ::-webkit-scrollbar {
            width: 12px;
            height: 12px;
        }
        
        ::-webkit-scrollbar-track {
            background: #1e1e1e;
            border: 2px solid #000;
        }
        
        ::-webkit-scrollbar-thumb {
            background: #3a3a3a;
            border: 2px solid #000;
        }
        
        ::-webkit-scrollbar-thumb:hover {
            background: #4a4a4a;
        }
        
        .loading-overlay {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.8);
            display: none;
            justify-content: center;
            align-items: center;
            z-index: 9999;
        }
        
        .loading-overlay.show {
            display: flex;
        }
        
        .loading-text {
            color: #5f5;
            font-size: 1.5em;
            font-weight: bold;
            text-shadow: 2px 2px 0 #000;
            animation: pulse 1.5s infinite;
        }
        
        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.5; }
        }
    </style>
</head>
<body>
    <div class="loading-overlay" id="loadingOverlay">
        <div class="loading-text">Loading...</div>
    </div>
    <div class="save-indicator" id="saveIndicator">Configuration saved!</div>
    
    <div class="container">
        <header>
            <h1>BBBoatHUD</h1>
            <p>Configure your HUD modules and settings</p>
        </header>
        
        <div class="tabs">
            <button class="tab active" onclick="switchTab(0)">Active Modules</button>
            <button class="tab" onclick="switchTab(1)">Add Modules</button>
            <button class="tab" onclick="switchTab(2)">Vanilla HUD</button>
        </div>
        
        <div class="tab-content active" id="tab-0">
            <div id="activeModules"></div>
        </div>
        
        <div class="tab-content" id="tab-1">
            <h2 style="margin-bottom: 20px;">Module Library</h2>
            <div id="moduleLibrary" class="module-library"></div>
        </div>
        
        <div class="tab-content" id="tab-2">
            <h2 style="margin-bottom: 20px;">Vanilla HUD Configuration</h2>
            <div id="vanillaConfig" class="vanilla-config"></div>
        </div>
    </div>
    
    <script>
        let config = null;
        let modules = null;
        let expandedModules = new Set();
        
        function switchTab(index) {
            document.querySelectorAll('.tab').forEach((tab, i) => {
                tab.classList.toggle('active', i === index);
            });
            document.querySelectorAll('.tab-content').forEach((content, i) => {
                content.classList.toggle('active', i === index);
            });
        }
        
        async function loadConfig() {
            try {
                const response = await fetch('http://localhost:8765/api/config');
                config = await response.json();
                renderActiveModules();
                renderVanillaConfig();
            } catch (error) {
                console.error('Failed to load config:', error);
            }
        }
        
        async function loadModules() {
            try {
                const response = await fetch('http://localhost:8765/api/modules');
                modules = await response.json();
                renderModuleLibrary();
            } catch (error) {
                console.error('Failed to load modules:', error);
            }
        }
        
        function showSaveIndicator() {
            const indicator = document.getElementById('saveIndicator');
            indicator.classList.add('show');
            setTimeout(() => {
                indicator.classList.remove('show');
            }, 2000);
        }
        
        async function savePlacements() {
            try {
                await fetch('http://localhost:8765/api/modules/placements', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify(config.modulePlacements)
                });
                showSaveIndicator();
            } catch (error) {
                console.error('Failed to save placements:', error);
                alert('Failed to save configuration');
            }
        }
        
        async function saveVanillaConfig() {
            try {
                await fetch('http://localhost:8765/api/vanilla', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify(config)
                });
                showSaveIndicator();
            } catch (error) {
                console.error('Failed to save vanilla config:', error);
                alert('Failed to save configuration');
            }
        }
        
        async function addModule(identifier) {
            const loadingOverlay = document.getElementById('loadingOverlay');
            
            try {
                // Show loading overlay
                loadingOverlay.classList.add('show');
                
                await fetch('http://localhost:8765/api/modules/add', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ identifier })
                });
                
                // Wait a bit for the server to process
                await new Promise(resolve => setTimeout(resolve, 100));
                
                await loadConfig();
                
                // Collapse all modules and expand only the newly added one (which is at the end, index length-1)
                expandedModules.clear();
                expandedModules.add(config.modulePlacements.length - 1);
                
                // Switch to active modules tab and render
                switchTab(0);
                renderActiveModules();
                
                // Scroll to top to see the newly added module
                const activeModulesContainer = document.getElementById('activeModules');
                activeModulesContainer.scrollTop = 0;
                
                showSaveIndicator();
            } catch (error) {
                console.error('Failed to add module:', error);
                alert('Failed to add module');
            } finally {
                // Hide loading overlay
                loadingOverlay.classList.remove('show');
            }
        }
        
        function removeModule(index) {
            // Collapse all modules when removing
            expandedModules.clear();
            
            config.modulePlacements.splice(index, 1);
            savePlacements();
            renderActiveModules();
        }
        
        function getFriendlyName(identifier) {
            // Extract filename from path like bb_boat_hud:bb_boat_hud_modules/boatlabs/scoreboard.lua
            const parts = identifier.split('/');
            const filename = parts[parts.length - 1];
            // Remove .lua extension
            const nameWithoutExt = filename.replace('.lua', '');
            // Replace underscores with spaces and capitalize each word
            return nameWithoutExt.split('_').map(word => 
                word.charAt(0).toUpperCase() + word.slice(1)
            ).join(' ');
        }
        
        function renderActiveModules() {
            const container = document.getElementById('activeModules');
            
            if (!config || config.modulePlacements.length === 0) {
                container.innerHTML = '<div class="empty-state"><div class="empty-state-icon">[_]</div><p>No modules added yet</p><p style="margin-top: 10px; font-size: 0.9em;">Go to "Add Modules" tab to add your first module</p></div>';
                return;
            }
            
            // Render in reverse order so newest (last in array) appears first
            const reversedPlacements = [...config.modulePlacements].reverse();
            
            container.innerHTML = reversedPlacements.map((placement, reverseIndex) => {
                // Calculate the actual index in the original array
                const index = config.modulePlacements.length - 1 - reverseIndex;
                
                return `
                <div class="module-item">
                    <div class="module-header" onclick="toggleModule(${index})">
                        <div>
                            <div class="module-title">${getFriendlyName(placement.identifier)}</div>
                            <div class="module-id">${placement.identifier}</div>
                        </div>
                        <span class="expand-indicator" id="indicator-${index}">${expandedModules.has(index) ? '-' : '+'}</span>
                    </div>
                    <div class="module-details ${expandedModules.has(index) ? 'expanded' : ''}" id="details-${index}">
                        <div class="form-group">
                            <label>Anchor Position</label>
                            <select onchange="updateAnchor(${index}, this.value)">
                                ${['TOP_LEFT', 'TOP_CENTER', 'TOP_RIGHT', 'MIDDLE_LEFT', 'MIDDLE_CENTER', 'MIDDLE_RIGHT', 'BOTTOM_LEFT', 'BOTTOM_CENTER', 'BOTTOM_RIGHT'].map(anchor => `
                                    <option value="${anchor}" ${placement.anchorType === anchor ? 'selected' : ''}>
                                        ${anchor.replace(/_/g, ' ')}
                                    </option>
                                `).join('')}
                            </select>
                        </div>
                        <div class="form-group">
                            <label>X Offset</label>
                            <div class="slider-input-group">
                                <input type="range" min="-300" max="300" value="${placement.xOffset}" 
                                       oninput="updateOffsetFromSlider(${index}, 'x', this.value)">
                                <input type="number" min="-300" max="300" value="${placement.xOffset}" 
                                       oninput="updateOffsetFromInput(${index}, 'x', this.value)">
                            </div>
                        </div>
                        <div class="form-group">
                            <label>Y Offset</label>
                            <div class="slider-input-group">
                                <input type="range" min="-300" max="300" value="${placement.yOffset}" 
                                       oninput="updateOffsetFromSlider(${index}, 'y', this.value)">
                                <input type="number" min="-300" max="300" value="${placement.yOffset}" 
                                       oninput="updateOffsetFromInput(${index}, 'y', this.value)">
                            </div>
                        </div>
                        <div class="form-group">
                            <label>Scale</label>
                            <div class="slider-input-group">
                                <input type="range" min="0.1" max="10" step="0.1" value="${placement.scale}" 
                                       oninput="updateScaleFromSlider(${index}, this.value)">
                                <input type="number" min="0.1" max="10" step="0.1" value="${placement.scale.toFixed(2)}" 
                                       oninput="updateScaleFromInput(${index}, this.value)">
                            </div>
                        </div>
                        <div class="form-group">
                            <label>Rotation (degrees)</label>
                            <div class="slider-input-group">
                                <input type="range" min="-180" max="180" step="1" value="${placement.angle}" 
                                       oninput="updateAngleFromSlider(${index}, this.value)">
                                <input type="number" min="-180" max="180" step="1" value="${placement.angle.toFixed(1)}" 
                                       oninput="updateAngleFromInput(${index}, this.value)">
                            </div>
                        </div>
                        <button class="danger" onclick="removeModule(${index})" style="width: 100%; margin-top: 10px;">Remove Module</button>
                    </div>
                </div>
            `}).join('');
        }
        
        function toggleModule(index) {
            if (expandedModules.has(index)) {
                // If clicking on already expanded module, collapse it
                expandedModules.delete(index);
            } else {
                // Collapse all others and expand this one
                expandedModules.clear();
                expandedModules.add(index);
            }
            
            // Re-render to update all indicators and states
            renderActiveModules();
        }
        
        function updateAnchor(index, anchor) {
            config.modulePlacements[index].anchorType = anchor;
            savePlacements();
        }
        
        function updateOffsetFromSlider(index, axis, value) {
            const intValue = parseInt(value);
            if (axis === 'x') {
                config.modulePlacements[index].xOffset = intValue;
            } else {
                config.modulePlacements[index].yOffset = intValue;
            }
            // Update the corresponding number input
            const sliderGroup = event.target.parentElement;
            const numberInput = sliderGroup.querySelector('input[type="number"]');
            numberInput.value = intValue;
            savePlacements();
        }
        
        function updateOffsetFromInput(index, axis, value) {
            const intValue = parseInt(value);
            if (axis === 'x') {
                config.modulePlacements[index].xOffset = intValue;
            } else {
                config.modulePlacements[index].yOffset = intValue;
            }
            // Update the corresponding slider
            const inputGroup = event.target.parentElement;
            const slider = inputGroup.querySelector('input[type="range"]');
            slider.value = intValue;
            savePlacements();
        }
        
        function updateScaleFromSlider(index, value) {
            const floatValue = parseFloat(value);
            config.modulePlacements[index].scale = floatValue;
            // Update the corresponding number input
            const sliderGroup = event.target.parentElement;
            const numberInput = sliderGroup.querySelector('input[type="number"]');
            numberInput.value = floatValue.toFixed(2);
            savePlacements();
        }
        
        function updateScaleFromInput(index, value) {
            const floatValue = parseFloat(value);
            config.modulePlacements[index].scale = floatValue;
            // Update the corresponding slider
            const inputGroup = event.target.parentElement;
            const slider = inputGroup.querySelector('input[type="range"]');
            slider.value = floatValue;
            savePlacements();
        }
        
        function updateAngleFromSlider(index, value) {
            const floatValue = parseFloat(value);
            config.modulePlacements[index].angle = floatValue;
            // Update the corresponding number input
            const sliderGroup = event.target.parentElement;
            const numberInput = sliderGroup.querySelector('input[type="number"]');
            numberInput.value = floatValue.toFixed(1);
            savePlacements();
        }
        
        function updateAngleFromInput(index, value) {
            const floatValue = parseFloat(value);
            config.modulePlacements[index].angle = floatValue;
            // Update the corresponding slider
            const inputGroup = event.target.parentElement;
            const slider = inputGroup.querySelector('input[type="range"]');
            slider.value = floatValue;
            savePlacements();
        }
        
        function renderModuleLibrary() {
            const container = document.getElementById('moduleLibrary');
            
            if (!modules) {
                container.innerHTML = '<p>Loading modules...</p>';
                return;
            }
            
            const tree = {};
            for (const [identifier, path] of Object.entries(modules)) {
                let current = tree;
                for (let i = 0; i < path.length - 1; i++) {
                    if (!current[path[i]]) {
                        current[path[i]] = {};
                    }
                    current = current[path[i]];
                }
                const lastName = path[path.length - 1];
                current[lastName] = identifier;
            }
            
            let folderIdCounter = 0;
            
            function renderTree(obj, level = 0) {
                let html = '';
                for (const [key, value] of Object.entries(obj)) {
                    if (typeof value === 'string') {
                        html += `<button class="module-add-btn" onclick="addModule('${value}')">${key}</button>`;
                    } else {
                        const folderId = 'folder-' + (folderIdCounter++);
                        html += `<div class="folder" style="margin-left: ${level * 20}px">
                            <div class="folder-header" onclick="toggleFolder('${folderId}')">
                                <span>${key}</span>
                                <span class="folder-indicator" id="indicator-${folderId}">+</span>
                            </div>
                            <div class="folder-content" id="${folderId}">
                                ${renderTree(value, level + 1)}
                            </div>
                        </div>`;
                    }
                }
                return html;
            }
            
            container.innerHTML = renderTree(tree);
        }
        
        function toggleFolder(folderId) {
            const content = document.getElementById(folderId);
            const indicator = document.getElementById('indicator-' + folderId);
            content.classList.toggle('expanded');
            indicator.textContent = content.classList.contains('expanded') ? '-' : '+';
        }
        
        function renderVanillaConfig() {
            const container = document.getElementById('vanillaConfig');
            
            if (!config) {
                container.innerHTML = '<p>Loading configuration...</p>';
                return;
            }
            
            const vanillaElements = [
                { key: 'hotbar', name: 'Hotbar' },
                { key: 'experienceBar', name: 'Experience Bar' },
                { key: 'experienceLevel', name: 'Experience Level' },
                { key: 'heldItemTooltip', name: 'Held Item Tooltip' },
                { key: 'armorBar', name: 'Armor Bar' },
                { key: 'healthBar', name: 'Health Bar' },
                { key: 'foodBar', name: 'Food Bar' },
                { key: 'oxygenBar', name: 'Oxygen Bar' },
                { key: 'bossBar', name: 'Boss Bar' },
                { key: 'overlayMessage', name: 'Action Bar Message' }
            ];
            
            container.innerHTML = vanillaElements.map(elem => `
                <div class="vanilla-item">
                    <h3>${elem.name}</h3>
                    <div class="form-group">
                        <div class="checkbox-group">
                            <input type="checkbox" id="${elem.key}_render" 
                                   ${config[elem.key].render ? 'checked' : ''}
                                   onchange="updateVanillaRender('${elem.key}', this.checked)">
                            <label for="${elem.key}_render">Render</label>
                        </div>
                    </div>
                    <div class="form-group">
                        <label>X Offset</label>
                        <input type="number" value="${config[elem.key].xOffset}" 
                               onchange="updateVanillaOffset('${elem.key}', 'xOffset', this.value)">
                    </div>
                    <div class="form-group">
                        <label>Y Offset</label>
                        <input type="number" value="${config[elem.key].yOffset}" 
                               onchange="updateVanillaOffset('${elem.key}', 'yOffset', this.value)">
                    </div>
                </div>
            `).join('');
        }
        
        function updateVanillaRender(key, value) {
            config[key].render = value;
            saveVanillaConfig();
        }
        
        function updateVanillaOffset(key, offset, value) {
            config[key][offset] = parseInt(value);
            saveVanillaConfig();
        }
        
        loadConfig();
        loadModules();
    </script>
</body>
</html>
        """;
    }
}
