let config = null;
let modules = null;
let expandedModules = new Set();

function switchTab(index) {
    document.querySelectorAll('.tab').forEach((tab, i) => {
        tab.classList.toggle('active', i === index);
    });
    document.querySelectorAll('.tab-content').forEach((content, i) => {
        content.classList.toggle('active', i === index);
    });
}

async function loadConfig() {
    try {
        const response = await fetch('http://localhost:8765/api/config');
        config = await response.json();
        renderActiveModules();
        renderVanillaConfig();
    } catch (error) {
        console.error('Failed to load config:', error);
    }
}

async function loadModules() {
    try {
        const response = await fetch('http://localhost:8765/api/modules');
        modules = await response.json();
        renderModuleLibrary();
    } catch (error) {
        console.error('Failed to load modules:', error);
    }
}

function showSaveIndicator() {
    const indicator = document.getElementById('saveIndicator');
    indicator.classList.add('show');
    setTimeout(() => {
        indicator.classList.remove('show');
    }, 2000);
}

async function savePlacements() {
    try {
        await fetch('http://localhost:8765/api/modules/placements', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(config.modulePlacements)
        });
        showSaveIndicator();
    } catch (error) {
        console.error('Failed to save placements:', error);
        alert('Failed to save configuration');
    }
}

async function saveVanillaConfig() {
    try {
        await fetch('http://localhost:8765/api/vanilla', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify(config)
        });
        showSaveIndicator();
    } catch (error) {
        console.error('Failed to save vanilla config:', error);
        alert('Failed to save configuration');
    }
}

async function addModule(identifier) {
    const loadingOverlay = document.getElementById('loadingOverlay');

    try {
        // Show loading overlay
        loadingOverlay.classList.add('show');

        await fetch('http://localhost:8765/api/modules/add', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ identifier })
        });

        // Wait a bit for the server to process
        await new Promise(resolve => setTimeout(resolve, 100));

        await loadConfig();

        // Collapse all modules and expand only the newly added one (which is at the end, index length-1)
        expandedModules.clear();
        expandedModules.add(config.modulePlacements.length - 1);

        // Switch to active modules tab and render
        switchTab(0);
        renderActiveModules();

        // Scroll to top to see the newly added module
        const activeModulesContainer = document.getElementById('activeModules');
        activeModulesContainer.scrollTop = 0;

        showSaveIndicator();
    } catch (error) {
        console.error('Failed to add module:', error);
        alert('Failed to add module');
    } finally {
        // Hide loading overlay
        loadingOverlay.classList.remove('show');
    }
}

function removeModule(index) {
    // Collapse all modules when removing
    expandedModules.clear();

    config.modulePlacements.splice(index, 1);
    savePlacements();
    renderActiveModules();
}

function getFriendlyName(identifier) {
    // Extract filename from path like bb_boat_hud:bb_boat_hud_modules/boatlabs/scoreboard.lua
    const parts = identifier.split('/');
    const filename = parts[parts.length - 1];
    // Remove .lua extension
    const nameWithoutExt = filename.replace('.lua', '');
    // Replace underscores with spaces and capitalize each word
    return nameWithoutExt.split('_').map(word =>
        word.charAt(0).toUpperCase() + word.slice(1)
    ).join(' ');
}

function renderActiveModules() {
    const container = document.getElementById('activeModules');

    if (!config || config.modulePlacements.length === 0) {
        container.innerHTML = '<div class="empty-state"><div class="empty-state-icon">[_]</div><p>No modules added yet</p><p style="margin-top: 10px; font-size: 0.9em;">Go to "Add Modules" tab to add your first module</p></div>';
        return;
    }

    // Render in reverse order so newest (last in array) appears first
    const reversedPlacements = [...config.modulePlacements].reverse();

    container.innerHTML = reversedPlacements.map((placement, reverseIndex) => {
        // Calculate the actual index in the original array
        const index = config.modulePlacements.length - 1 - reverseIndex;

        return `
        <div class="module-item">
            <div class="module-header" onclick="toggleModule(${index})">
                <div>
                    <div class="module-title">${getFriendlyName(placement.identifier)}</div>
                    <div class="module-id">${placement.identifier}</div>
                </div>
                <span class="expand-indicator" id="indicator-${index}">${expandedModules.has(index) ? '-' : '+'}</span>
            </div>
            <div class="module-details ${expandedModules.has(index) ? 'expanded' : ''}" id="details-${index}">
                <div class="form-group">
                    <label>Anchor Position</label>
                    <select onchange="updateAnchor(${index}, this.value)">
                        ${['TOP_LEFT', 'TOP_CENTER', 'TOP_RIGHT', 'MIDDLE_LEFT', 'MIDDLE_CENTER', 'MIDDLE_RIGHT', 'BOTTOM_LEFT', 'BOTTOM_CENTER', 'BOTTOM_RIGHT'].map(anchor => `
                            <option value="${anchor}" ${placement.anchorType === anchor ? 'selected' : ''}>
                                ${anchor.replace(/_/g, ' ')}
                            </option>
                        `).join('')}
                    </select>
                </div>
                <div class="form-group">
                    <label>X Offset</label>
                    <div class="slider-input-group">
                        <input type="range" min="-300" max="300" value="${placement.xOffset}" 
                                oninput="updateOffsetFromSlider(${index}, 'x', this.value)">
                        <input type="number" min="-300" max="300" value="${placement.xOffset}" 
                                oninput="updateOffsetFromInput(${index}, 'x', this.value)">
                    </div>
                </div>
                <div class="form-group">
                    <label>Y Offset</label>
                    <div class="slider-input-group">
                        <input type="range" min="-300" max="300" value="${placement.yOffset}" 
                                oninput="updateOffsetFromSlider(${index}, 'y', this.value)">
                        <input type="number" min="-300" max="300" value="${placement.yOffset}" 
                                oninput="updateOffsetFromInput(${index}, 'y', this.value)">
                    </div>
                </div>
                <div class="form-group">
                    <label>Scale</label>
                    <div class="slider-input-group">
                        <input type="range" min="0.1" max="10" step="0.1" value="${placement.scale}" 
                                oninput="updateScaleFromSlider(${index}, this.value)">
                        <input type="number" min="0.1" max="10" step="0.1" value="${placement.scale.toFixed(2)}" 
                                oninput="updateScaleFromInput(${index}, this.value)">
                    </div>
                </div>
                <div class="form-group">
                    <label>Rotation (degrees)</label>
                    <div class="slider-input-group">
                        <input type="range" min="-180" max="180" step="1" value="${placement.angle}" 
                                oninput="updateAngleFromSlider(${index}, this.value)">
                        <input type="number" min="-180" max="180" step="1" value="${placement.angle.toFixed(1)}" 
                                oninput="updateAngleFromInput(${index}, this.value)">
                    </div>
                </div>
                <button class="danger" onclick="removeModule(${index})" style="width: 100%; margin-top: 10px;">Remove Module</button>
            </div>
        </div>
    `}).join('');
}

function toggleModule(index) {
    if (expandedModules.has(index)) {
        // If clicking on already expanded module, collapse it
        expandedModules.delete(index);
    } else {
        // Collapse all others and expand this one
        expandedModules.clear();
        expandedModules.add(index);
    }

    // Re-render to update all indicators and states
    renderActiveModules();
}

function updateAnchor(index, anchor) {
    config.modulePlacements[index].anchorType = anchor;
    savePlacements();
}

function updateOffsetFromSlider(index, axis, value) {
    const intValue = parseInt(value);
    if (axis === 'x') {
        config.modulePlacements[index].xOffset = intValue;
    } else {
        config.modulePlacements[index].yOffset = intValue;
    }
    // Update the corresponding number input
    const sliderGroup = event.target.parentElement;
    const numberInput = sliderGroup.querySelector('input[type="number"]');
    numberInput.value = intValue;
    savePlacements();
}

function updateOffsetFromInput(index, axis, value) {
    const intValue = parseInt(value);
    if (axis === 'x') {
        config.modulePlacements[index].xOffset = intValue;
    } else {
        config.modulePlacements[index].yOffset = intValue;
    }
    // Update the corresponding slider
    const inputGroup = event.target.parentElement;
    const slider = inputGroup.querySelector('input[type="range"]');
    slider.value = intValue;
    savePlacements();
}

function updateScaleFromSlider(index, value) {
    const floatValue = parseFloat(value);
    config.modulePlacements[index].scale = floatValue;
    // Update the corresponding number input
    const sliderGroup = event.target.parentElement;
    const numberInput = sliderGroup.querySelector('input[type="number"]');
    numberInput.value = floatValue.toFixed(2);
    savePlacements();
}

function updateScaleFromInput(index, value) {
    const floatValue = parseFloat(value);
    config.modulePlacements[index].scale = floatValue;
    // Update the corresponding slider
    const inputGroup = event.target.parentElement;
    const slider = inputGroup.querySelector('input[type="range"]');
    slider.value = floatValue;
    savePlacements();
}

function updateAngleFromSlider(index, value) {
    const floatValue = parseFloat(value);
    config.modulePlacements[index].angle = floatValue;
    // Update the corresponding number input
    const sliderGroup = event.target.parentElement;
    const numberInput = sliderGroup.querySelector('input[type="number"]');
    numberInput.value = floatValue.toFixed(1);
    savePlacements();
}

function updateAngleFromInput(index, value) {
    const floatValue = parseFloat(value);
    config.modulePlacements[index].angle = floatValue;
    // Update the corresponding slider
    const inputGroup = event.target.parentElement;
    const slider = inputGroup.querySelector('input[type="range"]');
    slider.value = floatValue;
    savePlacements();
}

function renderModuleLibrary() {
    const container = document.getElementById('moduleLibrary');

    if (!modules) {
        container.innerHTML = '<p>Loading modules...</p>';
        return;
    }

    const tree = {};
    for (const [identifier, path] of Object.entries(modules)) {
        let current = tree;
        for (let i = 0; i < path.length - 1; i++) {
            if (!current[path[i]]) {
                current[path[i]] = {};
            }
            current = current[path[i]];
        }
        const lastName = path[path.length - 1];
        current[lastName] = identifier;
    }

    let folderIdCounter = 0;

    function renderTree(obj, level = 0) {
        let html = '';
        for (const [key, value] of Object.entries(obj)) {
            if (typeof value === 'string') {
                html += `<button class="module-add-btn" onclick="addModule('${value}')">${key}</button>`;
            } else {
                const folderId = 'folder-' + (folderIdCounter++);
                html += `<div class="folder" style="margin-left: ${level * 20}px">
                    <div class="folder-header" onclick="toggleFolder('${folderId}')">
                        <span>${key}</span>
                        <span class="folder-indicator" id="indicator-${folderId}">+</span>
                    </div>
                    <div class="folder-content" id="${folderId}">
                        ${renderTree(value, level + 1)}
                    </div>
                </div>`;
            }
        }
        return html;
    }

    container.innerHTML = renderTree(tree);
}

function toggleFolder(folderId) {
    const content = document.getElementById(folderId);
    const indicator = document.getElementById('indicator-' + folderId);
    content.classList.toggle('expanded');
    indicator.textContent = content.classList.contains('expanded') ? '-' : '+';
}

function renderVanillaConfig() {
    const container = document.getElementById('vanillaConfig');

    if (!config) {
        container.innerHTML = '<p>Loading configuration...</p>';
        return;
    }

    const vanillaElements = [
        { key: 'hotbar', name: 'Hotbar' },
        { key: 'experienceBar', name: 'Experience Bar' },
        { key: 'experienceLevel', name: 'Experience Level' },
        { key: 'heldItemTooltip', name: 'Held Item Tooltip' },
        { key: 'armorBar', name: 'Armor Bar' },
        { key: 'healthBar', name: 'Health Bar' },
        { key: 'foodBar', name: 'Food Bar' },
        { key: 'oxygenBar', name: 'Oxygen Bar' },
        { key: 'bossBar', name: 'Boss Bar' },
        { key: 'overlayMessage', name: 'Action Bar Message' }
    ];

    container.innerHTML = vanillaElements.map(elem => `
        <div class="vanilla-item">
            <h3>${elem.name}</h3>
            <div class="form-group">
                <div class="checkbox-group">
                    <input type="checkbox" id="${elem.key}_render" 
                            ${config[elem.key].render ? 'checked' : ''}
                            onchange="updateVanillaRender('${elem.key}', this.checked)">
                    <label for="${elem.key}_render">Render</label>
                </div>
            </div>
            <div class="form-group">
                <label>X Offset</label>
                <input type="number" value="${config[elem.key].xOffset}" 
                        onchange="updateVanillaOffset('${elem.key}', 'xOffset', this.value)">
            </div>
            <div class="form-group">
                <label>Y Offset</label>
                <input type="number" value="${config[elem.key].yOffset}" 
                        onchange="updateVanillaOffset('${elem.key}', 'yOffset', this.value)">
            </div>
        </div>
    `).join('');
}

function updateVanillaRender(key, value) {
    config[key].render = value;
    saveVanillaConfig();
}

function updateVanillaOffset(key, offset, value) {
    config[key][offset] = parseInt(value);
    saveVanillaConfig();
}

loadConfig();
loadModules();