package dev.overgrown.aspectslib.api;

import dev.overgrown.aspectslib.data.*;
import dev.overgrown.aspectslib.entity.aura_node.client.AuraNodeVisibilityConfig;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;
import org.jetbrains.annotations.Nullable;

import java.util.Optional;
import java.util.function.BiPredicate;
import net.minecraft.class_1657;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_2960;
import net.minecraft.class_7923;

/**
 * Public API for AspectsLib functionality.
 * <p>
 * Provides:
 * <li>Access to aspect data on items</li>
 * <li>Aspect lookup methods</li>
 * <li>Item-aspect registration</li>
 * </p>
 * <p>
* <br>
 * Usage Example:
 * <pre>{@code
 * // Get aspects from item
 * AspectData data = AspectsAPI.getAspectData(stack);
 * }</pre>
 * </p>
 * <p>
 * <pre>{@code
 * // Add aspect to item
 * AspectsAPI.addAspect(stack, new Identifier("mymod:ignis"), 5);
 * }</pre>
 * </p>
 * <p>
 * <pre>{@code
 * // Register default aspects for item
 * AspectsAPI.registerItemAspect(Items.DIAMOND, new Identifier("aspectslib:vitreus"), 10);
 * }</pre>
 * </p>
 */
public class AspectsAPI {

    /**
     * Gets the aspect data from an ItemStack
     * @param stack The ItemStack to check
     * @return The aspect data, or AspectData.DEFAULT if none
     */
    public static AspectData getAspectData(class_1799 stack) {
        return ((IAspectDataProvider) (Object) stack).aspectslib$getAspectData();
    }

    /**
     * Sets the aspect data on an ItemStack
     * @param stack The ItemStack to modify
     * @param data The aspect data to set, or null to clear
     */
    public static void setAspectData(class_1799 stack, @Nullable AspectData data) {
        ((IAspectDataProvider) (Object) stack).aspectslib$setAspectData(data);
    }

    /**
     * Adds aspects to an ItemStack
     * @param stack The ItemStack to modify
     * @param aspectId The identifier of the aspect to add
     * @param amount The amount to add
     * @return true if successful, false if aspect not found
     */
    public static boolean addAspect(class_1799 stack, class_2960 aspectId, int amount) {
        Aspect aspect = ModRegistries.ASPECTS.get(aspectId);
        if (aspect == null) {
            return false;
        }

        AspectData currentData = getAspectData(stack);
        AspectData.Builder builder = new AspectData.Builder(currentData);
        builder.add(aspectId, amount);
        setAspectData(stack, builder.build());
        return true;
    }

    /**
     * Adds aspects to an ItemStack by name
     * @param stack The ItemStack to modify
     * @param aspectName The name of the aspect to add
     * @param amount The amount to add
     * @return true if successful, false if aspect not found
     */
    public static boolean addAspectByName(class_1799 stack, String aspectName, int amount) {
        AspectData currentData = getAspectData(stack);
        AspectData.Builder builder = new AspectData.Builder(currentData);
        builder.addByName(aspectName, amount);
        setAspectData(stack, builder.build());
        return true;
    }

    /**
     * Registers default aspects for an items
     * @param item The items to register aspects for
     * @param aspectId The identifier of the aspect
     * @param amount The default amount
     */
    public static void registerItemAspect(class_1792 item, class_2960 aspectId, int amount) {
        class_2960 itemId = class_7923.field_41178.method_10221(item);
        Aspect aspect = ModRegistries.ASPECTS.get(aspectId);
        
        if (aspect != null) {
            Object2IntOpenHashMap<class_2960> aspects = new Object2IntOpenHashMap<>();
            aspects.put(aspectId, amount);
            ItemAspectRegistry.register(itemId, new AspectData(aspects));
        }
    }

    /**
     * Registers default aspects for an items by name
     * @param item The items to register aspects for
     * @param aspectName The name of the aspect
     * @param amount The default amount
     */
    public static void registerItemAspectByName(class_1792 item, String aspectName, int amount) {
        class_2960 itemId = class_7923.field_41178.method_10221(item);
        AspectData.Builder builder = new AspectData.Builder(AspectData.DEFAULT);
        builder.addByName(aspectName, amount);
        ItemAspectRegistry.register(itemId, builder.build());
    }

    /**
     * Gets an aspect by its identifier
     * @param aspectId The identifier of the aspect
     * @return The aspect, or empty if not found
     */
    public static Optional<Aspect> getAspect(class_2960 aspectId) {
        return Optional.ofNullable(ModRegistries.ASPECTS.get(aspectId));
    }

    /**
     * Gets an aspect by its name
     * @param aspectName The name of the aspect
     * @return The aspect, or empty if not found
     */
    public static Optional<Aspect> getAspectByName(String aspectName) {
        class_2960 aspectId = AspectManager.NAME_TO_ID.get(aspectName);
        return aspectId != null ? getAspect(aspectId) : Optional.empty();
    }

    /**
     * Creates a new AspectData builder
     * @return A new builder instance
     */
    public static AspectData.Builder createAspectDataBuilder() {
        return new AspectData.Builder(AspectData.DEFAULT);
    }

    /**
     * Gets all loaded aspects
     * @return A map of all loaded aspects
     */
    public static java.util.Map<class_2960, Aspect> getAllAspects() {
        return java.util.Collections.unmodifiableMap(ModRegistries.ASPECTS);
    }

    /**
     * Adds a condition for when aura nodes should be fully visible.
     * @param condition A predicate that takes a PlayerEntity and boolean indicating if the node has aspects
     */
    public static void addAuraNodeVisibilityCondition(BiPredicate<class_1657, Boolean> condition) {
        AuraNodeVisibilityConfig.addVisibilityCondition(condition);
    }

    /**
     * Sets whether aura nodes should always be fully visible.
     * @param alwaysShow true to always show nodes, false to use conditions
     */
    public static void setAuraNodesAlwaysVisible(boolean alwaysShow) {
        AuraNodeVisibilityConfig.setAlwaysShow(alwaysShow);
    }
}