package dev.overgrown.aspectslib.data;

import dev.overgrown.aspectslib.AspectsLib;
import dev.overgrown.aspectslib.api.IAspectDataProvider;
import dev.overgrown.aspectslib.codec.CodecUtils;
import com.mojang.serialization.Codec;
import com.mojang.serialization.codecs.RecordCodecBuilder;
import dev.overgrown.aspectslib.resonance.ResonanceCalculator;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;
import java.util.Collections;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import net.minecraft.class_2487;
import net.minecraft.class_2509;
import net.minecraft.class_2520;
import net.minecraft.class_2540;
import net.minecraft.class_2960;

/**
 * The `AspectData` class is the container for aspect amounts associated with an item and manages the association between items and their aspects. It stores the levels of aspects and provides functionality for NBT serialization, network synchronization, and manipulation. Now uses direct Identifier->int mapping instead of registry entries.
 * <p>
 * Features:
 * <li>NBT serialization</li>
 * <li>Network synchronization</li>
 * <li>Builder pattern for modification</li>
 * </p>
 * <p>
 * Usage:
 * <li>Stored on ItemStack via {@link IAspectDataProvider}</li>
 * <li>Used in tooltip rendering</li>
 * </p>
 */
public class AspectData {

    // Default instance with no aspects
    public static final AspectData DEFAULT = new AspectData(new Object2IntOpenHashMap<>());

    // Codec for serialization and deserialization
    private static Codec<Object2IntOpenHashMap<class_2960>> getInlineCodec() {
        return Codec.unboundedMap(class_2960.field_25139, Codec.INT)
                .xmap(Object2IntOpenHashMap::new, Function.identity());
    }

    private static Codec<AspectData> getBaseCodec() {
        return RecordCodecBuilder.create(instance ->
                instance.group(
                        getInlineCodec().fieldOf("aspects").forGetter(component -> component.aspects)
                ).apply(instance, AspectData::new)
        );
    }

    public static final Codec<AspectData> CODEC = CodecUtils.withAlternative(
            CodecUtils.lazy(AspectData::getBaseCodec),
            getInlineCodec().xmap(AspectData::new, data -> data.aspects)
    );

    // Internal storage for aspects and their levels - now uses Identifier instead of RegistryEntry
    private final Object2IntOpenHashMap<class_2960> aspects;

    // Constructor
    public AspectData(Object2IntOpenHashMap<class_2960> aspects) {
        this.aspects = aspects;
    }

    /**
     * Gets the level of the specified aspect by identifier.
     *
     * @param aspectId The aspect identifier to query.
     * @return The level of the aspect, or 0 if not present.
     */
    public int getLevel(class_2960 aspectId) {
        return this.aspects.getInt(aspectId);
    }

    /**
     * Gets the level of the specified aspect by name.
     *
     * @param aspectName The aspect name to query.
     * @return The level of the aspect, or 0 if not present.
     */
    public int getLevelByName(String aspectName) {
        class_2960 aspectId = AspectManager.NAME_TO_ID.get(aspectName);
        return aspectId != null ? this.aspects.getInt(aspectId) : 0;
    }

    /**
     * Gets the set of aspect identifiers.
     *
     * @return An unmodifiable set of aspect identifiers.
     */
    public Set<class_2960> getAspectIds() {
        return Collections.unmodifiableSet(this.aspects.keySet());
    }

    /**
     * Gets the aspect map with their levels.
     *
     * @return A map of aspect identifiers to their levels.
     */
    public Object2IntOpenHashMap<class_2960> getMap() {
        return this.aspects;
    }

    /**
     * Gets the number of registered aspects.
     *
     * @return The number of aspects.
     */
    public int getSize() {
        return this.aspects.size();
    }

    /**
     * Checks if there are no aspects registered.
     *
     * @return `true` if no aspects are registered, otherwise `false`.
     */
    public boolean isEmpty() {
        return this.aspects.isEmpty();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o instanceof AspectData other) {
            return this.aspects.equals(other.aspects);
        }
        return false;
    }

    @Override
    public String toString() {
        return "AspectData{aspects=" + this.aspects + "}";
    }

    /**
     * Adds aspects from another AspectData to this one.
     */
    public AspectData addAspect(AspectData aspectData) {
        for (Object2IntMap.Entry<class_2960> aspectIntegerEntry : aspectData.aspects.object2IntEntrySet()) {
            this.aspects.put(aspectIntegerEntry.getKey(), this.aspects.getOrDefault(aspectIntegerEntry.getKey(), 0) + aspectIntegerEntry.getIntValue());
        }
        return this;
    }

    /**
     * Writes this AspectData to NBT
     */
    public class_2487 toNbt() {
        class_2487 nbt = new class_2487();
        CODEC.encodeStart(class_2509.field_11560, this)
                .resultOrPartial(AspectsLib.LOGGER::error)
                .ifPresent(element -> nbt.method_10566("AspectData", element));
        return nbt;
    }

    /**
     * Reads AspectData from NBT
     */
    public static AspectData fromNbt(class_2487 nbt) {
        if (nbt.method_10573("AspectData", class_2520.field_33260)) {
            return CODEC.parse(class_2509.field_11560, nbt.method_10580("AspectData"))
                    .resultOrPartial(AspectsLib.LOGGER::error)
                    .orElse(DEFAULT);
        }
        return DEFAULT;
    }

    /**
     * Writes this AspectData to a packet buffer for network sync
     */
    public void toPacket(class_2540 buf) {
        buf.method_10804(aspects.size());
        for (Object2IntMap.Entry<class_2960> entry : aspects.object2IntEntrySet()) {
            buf.method_10812(entry.getKey());
            buf.method_10804(entry.getIntValue());
        }
    }

    /**
     * Reads AspectData from a packet buffer
     */
    public static AspectData fromPacket(class_2540 buf) {
        int size = buf.method_10816();
        Object2IntOpenHashMap<class_2960> aspects = new Object2IntOpenHashMap<>();
        for (int i = 0; i < size; i++) {
            class_2960 aspectId = buf.method_10810();
            int amount = buf.method_10816();
            aspects.put(aspectId, amount);
        }
        return new AspectData(aspects);
    }

    /**
     * Builder class for creating `AspectData` instances.
     */
    public static class Builder {
        private final Object2IntOpenHashMap<class_2960> aspects = new Object2IntOpenHashMap<>();

        public Builder(AspectData data) {
            this.aspects.putAll(data.aspects);
        }

        /**
         * Sets the level of an aspect by identifier.
         *
         * @param aspectId The aspect identifier to set.
         * @param level  The level to set.
         */
        public void set(class_2960 aspectId, int level) {
            if (level <= 0) {
                this.aspects.removeInt(aspectId);
            } else {
                this.aspects.put(aspectId, level);
            }
        }

        /**
         * Sets the level of an aspect by name.
         *
         * @param aspectName The aspect name to set.
         * @param level  The level to set.
         */
        public void setByName(String aspectName, int level) {
            class_2960 aspectId = AspectManager.NAME_TO_ID.get(aspectName);
            if (aspectId != null) {
                set(aspectId, level);
            }
        }

        /**
         * Adds a level to an aspect by identifier.
         *
         * @param aspectId The aspect identifier to add.
         * @param level  The level to add.
         */
        public void add(class_2960 aspectId, int level) {
            if (level > 0) {
                this.aspects.merge(aspectId, level, Integer::sum);
            }
        }

        /**
         * Adds a level to an aspect by name.
         *
         * @param aspectName The aspect name to add.
         * @param level  The level to add.
         */
        public void addByName(String aspectName, int level) {
            class_2960 aspectId = AspectManager.NAME_TO_ID.get(aspectName);
            if (aspectId != null) {
                this.aspects.put(aspectId, this.aspects.getOrDefault(aspectId, 0) + level);
            }
        }

        /**
         * Removes aspects that match the given predicate.
         *
         * @param predicate The predicate to test.
         */
        public void remove(Predicate<class_2960> predicate) {
            this.aspects.keySet().removeIf(predicate);
        }

        /**
         * Builds the `AspectData`.
         *
         * @return The constructed `AspectData`.
         */
        public AspectData build() {
            return new AspectData(this.aspects);
        }
    }

    /**
     * Calculates the Resonance Units (RU).
     */
    public double calculateTotalRU() {
        double total = 0;
        for (Object2IntMap.Entry<class_2960> entry : aspects.object2IntEntrySet()) {
            total += entry.getIntValue();
        }
        return total;
    }

    // Add resonance calculation
    public ResonanceCalculator.ResonanceResult calculateResonance() {
        return ResonanceCalculator.calculate(this);
    }
}