package dev.overgrown.aspectslib.data;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import dev.overgrown.aspectslib.AspectsLib;
import com.mojang.serialization.JsonOps;
import net.fabricmc.fabric.api.resource.IdentifiableResourceReloadListener;
import net.minecraft.class_2960;
import net.minecraft.class_3300;
import net.minecraft.class_3695;
import net.minecraft.class_4309;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * Simple map-based aspect loader that reloads from datapacks. (Loads aspect definitions from datapacks)
 * Much simpler than trying to use dynamic registries in 1.20.1!
 * <p>
 * Responsibilities:
 * <ol type="1">
 * <li>Loads aspect JSON files from data/aspectslib/aspects</li>
 * <li>Populates ModRegistries.ASPECTS</li>
 * </ol>
 * </p>
 * File format example (data/aspectslib/aspects/terra.json):
 * <pre>{@code
 * {
 *   "name": "Terra",
 *   "texture_location": "aspectslib:textures/aspects_icons/terra.png"
 * }
 * }</pre>
 */
public class AspectManager extends class_4309 implements IdentifiableResourceReloadListener {

    private static final Gson GSON = new Gson();

    /**
     * This parameter holds the Names of aspects against their registration Identifiers.
     */
    public static Map<String, class_2960> NAME_TO_ID = new HashMap<>();

    /**
     * Instantiate the AspectManager
     */
    public AspectManager() {
        super(GSON, "aspects");
    }

    /**
     * This method loads aspects from datapack JSON files into our simple map.
     * Gets called every time datapacks reload.
     */
    @Override
    protected void apply(Map<class_2960, JsonElement> prepared, class_3300 manager, class_3695 profiler) {
        // Clear existing data
        ModRegistries.ASPECTS.clear();
        NAME_TO_ID.clear();
        
        // Load aspects from datapack files
        AtomicInteger loadedCount = new AtomicInteger();
        for (Map.Entry<class_2960, JsonElement> entry : prepared.entrySet()) {
            class_2960 id = entry.getKey();
            JsonElement json = entry.getValue();

            Aspect.CODEC.parse(JsonOps.INSTANCE, json)
                    .resultOrPartial(error -> AspectsLib.LOGGER.error("Failed to parse aspect data {}: {}", id, error))
                    .ifPresent(aspect -> {
                        ModRegistries.ASPECTS.put(id, aspect);
                        NAME_TO_ID.put(aspect.name(), id);
                        loadedCount.getAndIncrement();
                        AspectsLib.LOGGER.debug("Loaded aspect: {} -> {}", aspect.name(), id);
                    });
        }

        AspectsLib.LOGGER.info("Loaded {} aspects from datapacks", loadedCount);
    }

    @Override
    public class_2960 getFabricId() {
        return AspectsLib.identifier("aspects");
    }
}