package dev.overgrown.aspectslib.data;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import dev.overgrown.aspectslib.AspectsLib;
import com.mojang.serialization.DataResult;
import com.mojang.serialization.JsonOps;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;
import net.fabricmc.fabric.api.resource.IdentifiableResourceReloadListener;
import net.minecraft.class_2960;
import net.minecraft.class_3300;
import net.minecraft.class_3518;
import net.minecraft.class_3695;
import net.minecraft.class_4309;
import net.minecraft.class_5699;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * This class populates the ItemAspectRegistry, with default integer values for different aspects and loads item-aspect associations from datapacks.
 * <p>
 * Responsibilities:
 * <ol type="1">
 * <li>Loads item tags from data/aspectslib/tags/items</li>
 * <li>Populates ItemAspectRegistry</li>
 * </ol>
 * </p>
 * Dependencies: Runs after AspectManager
 */
public class CustomItemTagManager extends class_4309 implements IdentifiableResourceReloadListener {

    private static final Gson GSON = new Gson();

    /**
     * Instantiate the CustomItemTagManager
     */
    public CustomItemTagManager() {
        super(GSON, "tags/items");
    }

    /**
     *  This method will look into the aspectslib:aspects items tag, and look for any objects that match the aspect
     *  json object, adding the correct data to the ItemAspectRegistry.
     */
    @Override
    protected void apply(Map<class_2960, JsonElement> prepared, class_3300 manager, class_3695 profiler) {
        ItemAspectRegistry.reset();

        if (ModRegistries.ASPECTS.isEmpty()) {
            AspectsLib.LOGGER.warn("Aspects map is empty, aspects may not be loaded yet");
            return;
        }

        AspectsLib.LOGGER.info("Processing {} items tag files", prepared.size());
        prepared.forEach((identifier, jsonElement) -> {
            AspectsLib.LOGGER.debug("Processing items tag: {}", identifier);
            if (identifier.equals(AspectsLib.identifier("aspects"))) {
                AspectsLib.LOGGER.info("Found aspects tag file, processing...");
                if (jsonElement instanceof JsonObject jsonObject) {
                    if (class_3518.method_15254(jsonObject, "replace") && class_3518.method_15270(jsonObject, "replace")) {
                        ItemAspectRegistry.reset();
                    }
                    if (class_3518.method_15264(jsonObject, "values")) {
                        JsonArray values = class_3518.method_15261(jsonObject, "values");
                        AspectsLib.LOGGER.info("Processing {} aspect entries", values.size());
                        for (JsonElement value : values) {
                            if (value instanceof JsonObject jsonItemObject) {
                                DataResult<class_5699.class_7476> id = class_5699.field_39274.parse(JsonOps.INSTANCE, jsonItemObject.get("id"));
                                if(id.result().isPresent()) {
                                    class_5699.class_7476 tagEntry = id.result().get();
                                    class_2960 itemId = tagEntry.comp_813();
                                    AspectsLib.LOGGER.debug("Processing entry: {} (tag: {})", itemId, tagEntry.comp_814());

                                    
                                    if (jsonItemObject.has("aspects")) {
                                        Object2IntOpenHashMap<class_2960> aspectAmount = new Object2IntOpenHashMap<>();
                                        JsonObject jsonAspectObject = jsonItemObject.getAsJsonObject("aspects");
                                        for (String aspectID : jsonAspectObject.asMap().keySet()) {
                                            class_2960 aspectIdentifier = new class_2960(aspectID);
                                            if (ModRegistries.ASPECTS.containsKey(aspectIdentifier)) {
                                                int amount = class_3518.method_15260(jsonAspectObject, aspectID);
                                                aspectAmount.put(aspectIdentifier, amount);
                                                AspectsLib.LOGGER.debug("Added aspect {} with amount {} to items {}", aspectID, amount, itemId);
                                            } else {
                                                AspectsLib.LOGGER.warn("Could not find aspect: {}", aspectID);
                                            }
                                        }
                                        
                                        if (!aspectAmount.isEmpty()) {
                                            ItemAspectRegistry.register(itemId, new AspectData(aspectAmount));
                                            AspectsLib.LOGGER.debug("Registered {} aspects for items: {}", aspectAmount.size(), itemId);
                                        }
                                    }
                                } else {
                                    AspectsLib.LOGGER.warn("Failed to parse items ID from: {}", jsonItemObject.get("id"));
                                }
                            }
                        }
                    }
                } else {
                    AspectsLib.LOGGER.warn("aspects.json is not a valid JSON object");
                }
            }
        });
        AspectsLib.LOGGER.info("Finished loading aspects from items tags. Registry contains {} items aspects.", ItemAspectRegistry.size());
    }

    @Override
    public class_2960 getFabricId() {
        return AspectsLib.identifier("item_aspects");
    }

    /**
     *  This method states that this should be run AFTER AspectManager.
     */
    @Override
    public Collection<class_2960> getFabricDependencies() {
        return List.of(AspectsLib.identifier("aspects"));
    }
}