package dev.overgrown.aspectslib.entity.aura_node;

import dev.overgrown.aspectslib.AspectsLib;
import dev.overgrown.aspectslib.data.AspectData;
import dev.overgrown.aspectslib.resonance.ResonanceCalculator;
import dev.overgrown.aspectslib.aether.DynamicAetherDensityManager;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;
import java.util.*;
import net.minecraft.class_1297;
import net.minecraft.class_1299;
import net.minecraft.class_1937;
import net.minecraft.class_1959;
import net.minecraft.class_2487;
import net.minecraft.class_2499;
import net.minecraft.class_2520;
import net.minecraft.class_2940;
import net.minecraft.class_2943;
import net.minecraft.class_2945;
import net.minecraft.class_2960;
import net.minecraft.class_5321;
import net.minecraft.class_5819;
import net.minecraft.class_6880;

public class AuraNodeEntity extends class_1297 {
    // Node types
    public enum NodeType {
        NORMAL, PURE, SINISTER, UNSTABLE, HUNGRY
    }

    // Regeneration accumulators
    private final Map<class_2960, Float> regenAccumulators = new HashMap<>();

    // Aspect state storage
    public static class AspectState {
        public int original;
        public int current;

        public AspectState(int original) {
            this.original = original;
            this.current = original;
        }

        public void regen(float amount) {
            if (current < original) {
                current = Math.min(original, current + (int)(original * amount));
            }
        }
    }

    // Tracked data
    private static final class_2940<Integer> NODE_TYPE = class_2945.method_12791(AuraNodeEntity.class, class_2943.field_13327);
    private static final class_2940<class_2487> ASPECTS_NBT = class_2945.method_12791(AuraNodeEntity.class, class_2943.field_13318);

    // Aspect identifiers
    public static final class_2960 FAMES_ASPECT = AspectsLib.identifier("fames");
    public static final class_2960 VITIUM_ASPECT = AspectsLib.identifier("vitium");

    private Map<class_2960, AspectState> aspects = new HashMap<>();
    private int instabilityCounter = 0;
    private int hungerCounter = 0;
    private int sinisterCounter = 0;

    public AuraNodeEntity(class_1299<?> type, class_1937 world) {
        super(type, world);
        this.field_5960 = true; // No collision
        this.method_5684(true); // Can't take damage
    }

    @Override
    protected void method_5693() {
        this.field_6011.method_12784(NODE_TYPE, NodeType.NORMAL.ordinal());
        this.field_6011.method_12784(ASPECTS_NBT, new class_2487());
    }

    @Override
    public void method_5773() {
        super.method_5773();

        if (!this.method_37908().method_8608()) {
            // Natural regeneration for aspects
            regenerateAspects();

            // Remove aspects that have been drained to 0
            removeDrainedAspects();

            // Check if node should die
            if (aspects.isEmpty()) {
//                this.discard();
                return;
            }

            // Type-specific behaviors
            switch (getNodeType()) {
                case SINISTER:
                    handleSinisterBehavior();
                    break;
                case HUNGRY:
                    handleHungryBehavior();
                    break;
                case UNSTABLE:
                    handleUnstableBehavior();
                    break;
            }

            // Update tracked data periodically
            if (this.field_6012 % 20 == 0) {
                updateTrackedAspects();
            }
        }
    }

    public int getRenderColour() {
        return 0xFFAA6655; // This a packed ARGB integer
    }

    private void regenerateAspects() {
        // Base regeneration rate (0.1% per second)
        final float BASE_REGEN_RATE = 0.00005f; // Much slower rate

        switch (getNodeType()) {
            case NORMAL, PURE -> regenerateStandard(BASE_REGEN_RATE);
            case SINISTER -> regenerateSinister(BASE_REGEN_RATE);
            case UNSTABLE -> regenerateUnstable(BASE_REGEN_RATE);
            case HUNGRY -> regenerateHungry(BASE_REGEN_RATE);
        }
    }

    private void regenerateStandard(float baseRate) {
        for (Map.Entry<class_2960, AspectState> entry : aspects.entrySet()) {
            regenerateAspect(entry.getKey(), entry.getValue(), baseRate);
        }
    }

    private void regenerateSinister(float baseRate) {
        for (Map.Entry<class_2960, AspectState> entry : aspects.entrySet()) {
            float rate = baseRate;
            if (entry.getKey().equals(VITIUM_ASPECT)) {
                rate *= 1.5f; // Vitium regenerates 50% faster
            }
            regenerateAspect(entry.getKey(), entry.getValue(), rate);
        }

        // Handle sinister corruption
        sinisterCounter++;
        if (sinisterCounter >= 100) { // Every 5 seconds
            sinisterCounter = 0;
            class_6880<class_1959> biomeEntry = method_37908().method_23753(method_24515());
            class_2960 biomeId = biomeEntry.method_40230().map(class_5321::method_29177).orElse(null);

            if (biomeId != null) {
                DynamicAetherDensityManager.addModification(
                        biomeId,
                        VITIUM_ASPECT,
                        10 // Add 10 Vitium per corruption cycle
                );
            }
        }
    }

    private void regenerateUnstable(float baseRate) {
        for (Map.Entry<class_2960, AspectState> entry : aspects.entrySet()) {
            regenerateAspect(entry.getKey(), entry.getValue(), baseRate * 0.8f);
        }
    }

    private void regenerateHungry(float baseRate) {
        AspectState famesState = aspects.get(FAMES_ASPECT);
        if (famesState == null) return;

        // If Fames isn't full, consume other aspects
        if (famesState.current < famesState.original) {
            float totalConsumed = 0;
            List<class_2960> toRemove = new ArrayList<>();

            for (Map.Entry<class_2960, AspectState> entry : aspects.entrySet()) {
                if (entry.getKey().equals(FAMES_ASPECT)) continue;

                AspectState state = entry.getValue();
                if (state.current <= 0) {
                    toRemove.add(entry.getKey());
                    continue;
                }

                // Calculate consumption (0.2% per tick)
                float consumeAmount = state.original * baseRate * 4;
                float available = Math.min(consumeAmount, state.current);

                state.current -= (int) available;
                totalConsumed += available;

                if (state.current <= 0) {
                    toRemove.add(entry.getKey());
                }
            }

            // Add consumed aspects to Fames
            if (totalConsumed > 0) {
                famesState.current = (int) Math.min(
                        famesState.original,
                        famesState.current + totalConsumed
                );
            }

            // Remove consumed aspects
            for (class_2960 id : toRemove) {
                aspects.remove(id);
            }
        } else {
            // If Fames is full, consume other aspects aggressively
            float totalConsumed = 0;
            List<class_2960> toRemove = new ArrayList<>();

            for (Map.Entry<class_2960, AspectState> entry : aspects.entrySet()) {
                if (entry.getKey().equals(FAMES_ASPECT)) continue;

                AspectState state = entry.getValue();
                if (state.current <= 0) {
                    toRemove.add(entry.getKey());
                    continue;
                }

                // More aggressive consumption (0.4% per tick)
                float consumeAmount = state.original * baseRate * 8;
                float available = Math.min(consumeAmount, state.current);

                state.current -= (int) available;
                totalConsumed += available;

                if (state.current <= 0) {
                    toRemove.add(entry.getKey());
                }
            }

            // Add consumed aspects to Fames (making it stronger)
            if (totalConsumed > 0) {
                famesState.original += (int) totalConsumed; // Make Fames stronger
                famesState.current = famesState.original;
            }

            // Remove consumed aspects
            for (class_2960 id : toRemove) {
                aspects.remove(id);
            }
        }

        // If only Fames remains, and it's full, start consuming itself
        if (aspects.size() == 1 && aspects.containsKey(FAMES_ASPECT) &&
                famesState.current >= famesState.original) {

            // Self-consumption (0.1% per tick)
            float consumeAmount = famesState.original * baseRate * 2;
            famesState.current = Math.max(0, famesState.current - (int) consumeAmount);

            if (famesState.current <= 0) {
                aspects.remove(FAMES_ASPECT);
            }
        }
    }

    private void regenerateAspect(class_2960 aspectId, AspectState state, float rate) {
        if (state.current >= state.original) return;

        // Get or create accumulator
        float accumulator = regenAccumulators.getOrDefault(aspectId, 0f);

        // Add this tick's regeneration
        float regenThisTick = state.original * rate;
        accumulator += regenThisTick;

        // Convert accumulated value to integer
        int toAdd = (int) accumulator;
        if (toAdd > 0) {
            int newCurrent = state.current + toAdd;
            if (newCurrent > state.original) {
                toAdd = state.original - state.current;
                newCurrent = state.original;
            }

            state.current = newCurrent;
            accumulator -= toAdd;

            // Debug logging
            AspectsLib.LOGGER.debug("Regenerated {}: {}/{} (+{})",
                    aspectId, state.current, state.original, toAdd);
        }

        // Store remaining fraction
        regenAccumulators.put(aspectId, accumulator);
    }

    private void removeDrainedAspects() {
        List<class_2960> toRemove = new ArrayList<>();
        for (Map.Entry<class_2960, AspectState> entry : aspects.entrySet()) {
            if (entry.getValue().current <= 0) {
                toRemove.add(entry.getKey());
            }
        }
        for (class_2960 id : toRemove) {
            aspects.remove(id);
        }
    }

    private void handleSinisterBehavior() {
        sinisterCounter++;
        if (sinisterCounter >= 100) { // Every 5 seconds
            sinisterCounter = 0;
            class_6880<class_1959> biomeEntry = method_37908().method_23753(method_24515());
            class_2960 biomeId = biomeEntry.method_40230().map(class_5321::method_29177).orElse(null);

            if (biomeId != null) {
                DynamicAetherDensityManager.addModification(
                        biomeId,
                        VITIUM_ASPECT,
                        10 // Add 10 Vitium per corruption cycle
                );
                AspectsLib.LOGGER.debug("Sinister node corrupting biome {} at {}", biomeId, this.method_24515());
            }
        }
    }

    private void handleHungryBehavior() {
        hungerCounter++;

        // Consume every 10 seconds (200 ticks)
        if (hungerCounter >= 200) {
            hungerCounter = 0;
            class_6880<class_1959> biomeEntry = method_37908().method_23753(method_24515());
            class_2960 biomeId = biomeEntry.method_40230().map(class_5321::method_29177).orElse(null);

            // Phase 1: Consume other aspects in the node
            if (aspects.size() > 1) {
                List<class_2960> nonFamesAspects = new ArrayList<>();
                for (class_2960 id : aspects.keySet()) {
                    if (!id.equals(FAMES_ASPECT)) {
                        nonFamesAspects.add(id);
                    }
                }

                if (!nonFamesAspects.isEmpty()) {
                    class_2960 target = nonFamesAspects.get(method_37908().method_8409().method_43048(nonFamesAspects.size()));
                    AspectState state = aspects.get(target);

                    // Consume 10% of the aspect
                    int consumeAmount = Math.max(1, state.current / 10);
                    state.current -= consumeAmount;

                    // If completely drained, remove with 10% chance
                    if (state.current <= 0 && method_37908().method_8409().method_43057() < 0.1f) {
                        aspects.remove(target);
                    }
                }
            }
            // Phase 2: Consume from environment
            else if (aspects.containsKey(FAMES_ASPECT)) {
                if (biomeId != null) {
                    // Drain 5 from all aspects in the biome
                    DynamicAetherDensityManager.drainAllAspects(biomeId, 5);
                    AspectsLib.LOGGER.debug("Hungry node draining environment at {}", this.method_24515());
                }

                AspectState famesState = aspects.get(FAMES_ASPECT);
                if (famesState.current > 0) {
                    famesState.current = Math.max(0, famesState.current - 10);
                    if (famesState.current <= 0) {
                        this.method_31472();
                    }
                }
            }
        }
    }

    private void handleUnstableBehavior() {
        // Check for opposing resonance every 5 seconds (100 ticks)
        if (this.field_6012 % 100 == 0) {
            // Create AspectData for resonance calculation
            Object2IntOpenHashMap<class_2960> aspectMap = new Object2IntOpenHashMap<>();
            for (Map.Entry<class_2960, AspectState> entry : aspects.entrySet()) {
                aspectMap.put(entry.getKey(), entry.getValue().current);
            }

            ResonanceCalculator.ResonanceResult result = ResonanceCalculator.calculate(new AspectData(aspectMap));

            // Increase instability if there's barrier cost (opposing resonance)
            if (result.barrierCost() > 0) {
                instabilityCounter += (int) Math.ceil(result.barrierCost());
                AspectsLib.LOGGER.debug("Unstable node instability: {}", instabilityCounter);

                // Explode when instability reaches threshold
                if (instabilityCounter >= 100) {
                    method_37908().method_8537(this, method_23317(), method_23318(), method_23321(), 3.0f, false, class_1937.class_7867.field_40888);
                    this.method_31472();
                }
            }
        }
    }

    private void updateTrackedAspects() {
        class_2487 aspectsNbt = new class_2487();
        for (Map.Entry<class_2960, AspectState> entry : aspects.entrySet()) {
            class_2487 stateNbt = new class_2487();
            stateNbt.method_10569("Original", entry.getValue().original);
            stateNbt.method_10569("Current", entry.getValue().current);
            aspectsNbt.method_10566(entry.getKey().toString(), stateNbt);
        }
        this.field_6011.method_12778(ASPECTS_NBT, aspectsNbt);
    }

    @Override
    public void method_5749(class_2487 nbt) {
        // Read node type
        this.setNodeType(NodeType.values()[nbt.method_10571("NodeType")]);

        // Read aspects - Create AspectState with correct current value
        aspects.clear();
        class_2499 aspectsList = nbt.method_10554("Aspects", class_2520.field_33260);
        for (class_2520 element : aspectsList) {
            class_2487 aspectNbt = (class_2487) element;
            class_2960 id = new class_2960(aspectNbt.method_10558("Id"));
            int original = aspectNbt.method_10550("Original");
            int current = aspectNbt.method_10550("Current");

            // Create AspectState with proper current value
            AspectState state = new AspectState(original);
            state.current = current;
            aspects.put(id, state);
        }

        instabilityCounter = nbt.method_10550("Instability");
        hungerCounter = nbt.method_10550("HungerCounter");
    }

    @Override
    public void method_5652(class_2487 nbt) {
        nbt.method_10567("NodeType", (byte) getNodeType().ordinal());

        class_2499 aspectsList = new class_2499();
        for (Map.Entry<class_2960, AspectState> entry : aspects.entrySet()) {
            class_2487 aspectNbt = new class_2487();
            aspectNbt.method_10582("Id", entry.getKey().toString());
            aspectNbt.method_10569("Original", entry.getValue().original);
            aspectNbt.method_10569("Current", entry.getValue().current);
            aspectsList.add(aspectNbt);
        }
        nbt.method_10566("Aspects", aspectsList);

        nbt.method_10569("Instability", instabilityCounter);
        nbt.method_10569("HungerCounter", hungerCounter);
    }

    protected void readCustomDataFromTrackedData() {
        this.setNodeType(NodeType.values()[field_6011.method_12789(NODE_TYPE)]);

        // Update aspects from tracked data
        class_2487 aspectsNbt = field_6011.method_12789(ASPECTS_NBT);
        aspects.clear();
        for (String key : aspectsNbt.method_10541()) {
            class_2960 id = new class_2960(key);
            class_2487 stateNbt = aspectsNbt.method_10562(key);
            int original = stateNbt.method_10550("Original");
            int current = stateNbt.method_10550("Current");
            aspects.put(id, new AspectState(original) {{
                this.current = current;
            }});
        }
    }

    // Getters and setters
    public NodeType getNodeType() {
        return NodeType.values()[field_6011.method_12789(NODE_TYPE)];
    }

    public void setNodeType(NodeType type) {
        field_6011.method_12778(NODE_TYPE, type.ordinal());
    }

    public Map<class_2960, AspectState> getAspects() {
        return Collections.unmodifiableMap(aspects);
    }

    public void setAspects(Map<class_2960, AspectState> aspects) {
        this.aspects = new HashMap<>(aspects);
        updateTrackedAspects();
    }

    public void initializeAspects(class_5819 random) {
        aspects.clear();

        switch (getNodeType()) {
            case PURE:
                // Pure node has only one aspect
                class_2960 aspect = getRandomAspect(random, 0.95f); // 95% chance for primal
                aspects.put(aspect, new AspectState(random.method_43048(100) + 50));
                break;

            case HUNGRY:
                // Hungry node always has Fames
                aspects.put(FAMES_ASPECT, new AspectState(random.method_43048(100) + 50));

                // 50% chance to have one additional aspect
                if (random.method_43056()) {
                    aspects.put(getRandomAspect(random, 0.9f), new AspectState(random.method_43048(100) + 50));
                }
                break;

            default:
                // Normal, Sinister, Unstable: 1-4 aspects
                int count = random.method_43048(4) + 1;
                for (int i = 0; i < count; i++) {
                    class_2960 newAspect = getRandomAspect(random, 0.8f); // 80% chance for primal
                    aspects.put(newAspect, new AspectState(random.method_43048(100) + 50));
                }
        }

        updateTrackedAspects();
    }

    private class_2960 getRandomAspect(class_5819 random, float primalChance) {
        // Organize aspects by tier
        List<class_2960> primal = Arrays.asList(
                AspectsLib.identifier("aer"), AspectsLib.identifier("aqua"), AspectsLib.identifier("ignis"),
                AspectsLib.identifier("ordo"), AspectsLib.identifier("perditio"), AspectsLib.identifier("terra")
        );

        List<class_2960> secondary = Arrays.asList(
                AspectsLib.identifier("gelum"), AspectsLib.identifier("lux"), AspectsLib.identifier("metallum"),
                AspectsLib.identifier("mortuus"), AspectsLib.identifier("motus"), AspectsLib.identifier("permutatio"),
                AspectsLib.identifier("potentia"), AspectsLib.identifier("vacuos"), AspectsLib.identifier("victus"),
                AspectsLib.identifier("vitreus")
        );

        List<class_2960> tertiary = Arrays.asList(
                AspectsLib.identifier("bestia"), AspectsLib.identifier("fames"), AspectsLib.identifier("exanimis"),
                AspectsLib.identifier("herba"), AspectsLib.identifier("instrumentum"), AspectsLib.identifier("praecantatio"),
                AspectsLib.identifier("spiritus"), AspectsLib.identifier("tenebrae"), AspectsLib.identifier("vinculum"),
                AspectsLib.identifier("volatus")
        );

        List<class_2960> quaternary = Arrays.asList(
                AspectsLib.identifier("alienis"), AspectsLib.identifier("alkimia"), AspectsLib.identifier("auram"),
                AspectsLib.identifier("aversion"), AspectsLib.identifier("cognitio"), AspectsLib.identifier("desiderium"),
                AspectsLib.identifier("fabrico"), AspectsLib.identifier("humanus"), AspectsLib.identifier("machina"),
                AspectsLib.identifier("praemunio"), AspectsLib.identifier("sensus"), AspectsLib.identifier("vitium")
        );

        // Safety checks for empty lists
        if (random.method_43057() < primalChance && !primal.isEmpty()) {
            return primal.get(random.method_43048(primal.size()));
        }

        float roll = random.method_43057();
        if (roll < 0.6f && !secondary.isEmpty()) {
            return secondary.get(random.method_43048(secondary.size()));
        } else if (roll < 0.9f && !tertiary.isEmpty()) {
            return tertiary.get(random.method_43048(tertiary.size()));
        } else if (!quaternary.isEmpty()) {
            return quaternary.get(random.method_43048(quaternary.size()));
        }

        // Fallback to primal if nothing else available
        return !primal.isEmpty() ? primal.get(random.method_43048(primal.size())) : AspectsLib.identifier("aer");
    }

    @Override
    public boolean method_5862() {
        return false; // Don't render fire
    }
}