package dev.overgrown.aspectslib.mixin;

import dev.overgrown.aspectslib.api.IAspectDataProvider;
import dev.overgrown.aspectslib.data.AspectData;
import dev.overgrown.aspectslib.data.ItemAspectRegistry;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;
import org.spongepowered.asm.mixin.Mixin;
import org.spongepowered.asm.mixin.Shadow;
import org.spongepowered.asm.mixin.Unique;
import org.spongepowered.asm.mixin.injection.At;
import org.spongepowered.asm.mixin.injection.Inject;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfo;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfoReturnable;

import java.util.Map;
import java.util.Objects;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_2487;
import net.minecraft.class_2960;
import net.minecraft.class_6862;
import net.minecraft.class_7923;

/**
 * Mixin to add aspect data support to ItemStack.
 * <p>
 * Responsibilities:
 * <ol type="1">
 * <li>Attaches AspectData to ItemStack via NBT</li>
 * <li>Handles initialization from tags/registry</li>
 * <li>Manages data copying and cache invalidation</li>
 * </ol>
 * <p>
 * Important Connections:
 * <li>{@link IAspectDataProvider}: Interface implemented by this mixin</li>
 * <li>{@link ItemAspectRegistry}: Source of default item aspects</li>
 * <li>{@link AspectData}: Actual aspect storage</li>
 */

@Mixin(class_1799.class)
public abstract class ItemStackMixin implements IAspectDataProvider {

    @Shadow public abstract class_1792 getItem();
    @Shadow public abstract class_2487 getOrCreateNbt();
    @Shadow public abstract class_2487 getNbt();

    @Unique
    private AspectData aspectslib$cachedAspectData = null;

    @Unique
    private boolean aspectslib$aspectDataInitialized = false;

    @Override
    public AspectData aspectslib$getAspectData() {
        if (!aspectslib$aspectDataInitialized) {
            aspectslib$initializeAspectData();
            aspectslib$aspectDataInitialized = true;
        }

        return Objects.requireNonNullElse(aspectslib$cachedAspectData, AspectData.DEFAULT);
    }

    @Override
    public void aspectslib$setAspectData(AspectData data) {
        aspectslib$cachedAspectData = data;
        aspectslib$aspectDataInitialized = true;

        // Handle NBT persistence (Null-safe data handling)
        class_2487 nbt = getNbt();
        if (data == null) {
            if (nbt != null) {
                nbt.method_10551("AspectsLibData");
                if (nbt.method_33133()) {
                    ((class_1799) (Object) this).method_7980(null);
                }
            }
        } else if (!data.isEmpty()) {
            getOrCreateNbt().method_10566("AspectsLibData", data.toNbt());
        }
    }

    /** Initialize aspect data from NBT or registry defaults */
    @Unique
    private void aspectslib$initializeAspectData() {
        class_2487 nbt = getNbt();
        if (nbt != null && nbt.method_10545("AspectsLibData")) {
            aspectslib$cachedAspectData = AspectData.fromNbt(nbt.method_10562("AspectsLibData"));
            return;
        }

        // Build from registry defaults
        AspectData aspectData = new AspectData(new Object2IntOpenHashMap<>());
        class_2960 itemId = class_7923.field_41178.method_10221(getItem());
        class_1799 self = (class_1799) (Object) this; // Use self for registry entry

        // Direct item registration
        if (ItemAspectRegistry.contains(itemId)) {
            aspectData = aspectData.addAspect(ItemAspectRegistry.get(itemId));
        }

        // Tag-based registrations
        for (Map.Entry<class_2960, AspectData> entry : ItemAspectRegistry.entries()) {
            class_2960 id = entry.getKey();
            AspectData itemAspectData = entry.getValue();

            // Exact match
            if (itemId.equals(id)) {
                aspectData = aspectData.addAspect(itemAspectData);
            }

            // Tag match
            class_6862<class_1792> tagKey = class_6862.method_40092(class_7923.field_41178.method_30517(), id);
            if (self.method_41409().method_40220(tagKey)) {
                aspectData = aspectData.addAspect(itemAspectData);
            }
        }

        aspectslib$cachedAspectData = aspectData.isEmpty() ? null : aspectData;
    }

    /** Reset cache when NBT changes */
    @Inject(method = "setNbt", at = @At("RETURN"))
    private void onSetNbt(class_2487 nbt, CallbackInfo ci) {
        aspectslib$aspectDataInitialized = false;
        aspectslib$cachedAspectData = null;
    }

    /** Copy aspect data when item is copied */
    @Inject(method = "copy", at = @At("RETURN"))
    private void onCopy(CallbackInfoReturnable<class_1799> cir) {
        class_1799 copy = cir.getReturnValue();
        if (aspectslib$cachedAspectData != null && !aspectslib$cachedAspectData.isEmpty()) {
            IAspectDataProvider copyProvider = (IAspectDataProvider) (Object) copy;
            if (copyProvider != null) {
                copyProvider.aspectslib$setAspectData(aspectslib$cachedAspectData);
            }
        }
    }
}