package dev.overgrown.aspectslib.networking;

import dev.overgrown.aspectslib.AspectsLib;
import dev.overgrown.aspectslib.data.Aspect;
import dev.overgrown.aspectslib.data.AspectManager;
import dev.overgrown.aspectslib.data.ModRegistries;
import net.fabricmc.fabric.api.networking.v1.PacketByteBufs;
import net.fabricmc.fabric.api.networking.v1.ServerPlayNetworking;
import net.minecraft.class_2540;
import net.minecraft.class_2960;
import net.minecraft.class_3222;
import java.util.HashMap;
import java.util.Map;

/**
 * This packet synchronizes both the aspect name mappings and the actual aspect data from server to client.
 * <p>
 * Responsibilities:
 * <ol type="1">
 *     <li>Serialize/deserialize aspect data for network transmission</li>
 *     <li>Send aspect registry to clients/li>
 * </ol>
 * </p>
 * <p>
 * Usage:
 * <li>Called during player join (see AspectsLib.java)</li>
 * <li>Used internally by library</li>
 * </p>
 * <br>
 * <p>
 * Important Connections:
 * <li>{@link AspectsLib}: Main mod class triggers sending</li>
 * <li>{@link ModRegistries}: Source of aspect data</li>
 * <li>{@link Aspect}: Aspect serialization logic</li>
 * </p>
 */
public class SyncAspectIdentifierPacket {
    public static final class_2960 ID = AspectsLib.identifier("sync_aspect_packet");

    /**
     * Writes the name-to-ID mapping to the buffer
     */
    public static void writeNameMap(class_2540 buf, Map<String, class_2960> nameMap) {
        buf.writeInt(nameMap.size());
        for (Map.Entry<String, class_2960> entry : nameMap.entrySet()) {
            buf.method_10814(entry.getKey());
            buf.method_10812(entry.getValue());
        }
    }

    /**
     * Writes the full aspect data to the buffer
     */
    public static void writeAspectData(class_2540 buf, Map<class_2960, Aspect> aspectMap) {
        buf.writeInt(aspectMap.size());
        for (Map.Entry<class_2960, Aspect> entry : aspectMap.entrySet()) {
            buf.method_10812(entry.getKey());
            Aspect.PACKET_CODEC.encode(buf, entry.getValue());
        }
    }

    /**
     * Writes both the name mapping and aspect data to the buffer
     */
    public static void writeFullData(class_2540 buf, Map<String, class_2960> nameMap, Map<class_2960, Aspect> aspectMap) {
        writeNameMap(buf, nameMap);
        writeAspectData(buf, aspectMap);
    }

    /**
     * Reads the name-to-ID mapping from the buffer
     */
    public static Map<String, class_2960> readNameMap(class_2540 buf) {
        int size = buf.readInt();
        Map<String, class_2960> map = new HashMap<>();
        for (int i = 0; i < size; i++) {
            String key = buf.method_19772();
            class_2960 value = buf.method_10810();
            map.put(key, value);
        }
        return map;
    }

    /**
     * Reads the aspect data from the buffer
     */
    public static Map<class_2960, Aspect> readAspectData(class_2540 buf) {
        int size = buf.readInt();
        Map<class_2960, Aspect> map = new HashMap<>();
        for (int i = 0; i < size; i++) {
            class_2960 id = buf.method_10810();
            Aspect aspect = Aspect.PACKET_CODEC.decode(buf);
            map.put(id, aspect);
        }
        return map;
    }

    /**
     * Sends the full aspect data (names + aspects) from server to client (Send full data to specific player)
     */
    public static void sendFullData(class_3222 player, Map<String, class_2960> nameMap, Map<class_2960, Aspect> aspectMap) {
        class_2540 buf = PacketByteBufs.create();
        writeFullData(buf, nameMap, aspectMap);
        ServerPlayNetworking.send(player, ID, buf);
    }

    /**
     * Sends all current aspect data from server to client (Send current registry data to player)
     */
    public static void sendAllData(class_3222 player) {
        AspectsLib.LOGGER.debug("Sending {} aspects and {} name mappings to client", 
                ModRegistries.ASPECTS.size(), AspectManager.NAME_TO_ID.size());
        sendFullData(player, AspectManager.NAME_TO_ID, ModRegistries.ASPECTS);
    }
}